/**
 * smartling-sdk
 * https://github.com/hightail/smartling-sdk
 *
 * Javascript SDK for Smartling. All functions are promise based using 'q' npm package
 *
 * Copyright (c) 2014 Hightail
 * Author: Justin Fiedler, Heavily modified by Vidhur Vohra
 *
 * Date: 1/15/14
 */

var request = require('request'),
    Q = require('q'),
    _ = require('lodash');

/**
 * Returns a search (GET var) string based on the @jsonObject
 * Handles nested objects using dot notation
 *
 * ex:
 * {
 *    myParam: 'something',
 *    myOtherParam: {
 *      somethingElse: someValue
 *    }
 * }
 *
 * returns:
 * myParam=something&myOtherParam.somethingElse=someValue
 *
 *
 * @param jsonObject
 * @returns {string}
 */
var jsonToSearchParameterString = function(jsonObject) {
  var getParams = [];


  function _jsonToSearchParameterString(_jsonObject, prefix) {
    //loop over all keys in the object
    _.each(_jsonObject, function(value, key) {
      if (_.isObject(value)) {
        //if the value is an object recurse
        _jsonToSearchParameterString(value, key + '.');
      } else {
        //if the value is not an object then add it to the GET params
        getParams.push(prefix + key + '=' + encodeURIComponent(value));
      }
    });
  }

  _jsonToSearchParameterString(jsonObject, '');

  return getParams.join('&');
};

function handleSmartlingResponse(response, deferred) {
  var smartlingResponse = response.response;
  if (smartlingResponse && smartlingResponse.code && smartlingResponse.code === 'SUCCESS') {
    deferred.resolve(smartlingResponse.data);
  } else {
    deferred.reject(response);
  }
}

function getStandardSmartlingRequestHandler(deferred) {
  return function (error, response, body) {
    if (!error && response.statusCode === 200) {
      var data = body;
      if (_.isString(data)) {
        try {
          data = JSON.parse(body);
        } catch (err) {

        }
      }
      handleSmartlingResponse(data, deferred);
    } else {
      if (!error || !_.isError(error)) {
        var errMsg = "\nResponse Code: " + response.statusCode;
        errMsg += "\nResponse Body: " + ((typeof body === 'string' && body) || JSON.stringify(body, null, 2) || JSON.stringify(response.body) || '').toString();
        // If the error object does not exist, the failure is most likely straight from Smartling. So printing the body.
        error = new Error(errMsg);
      }

      deferred.reject(error);
    }
  };
}

/**
 * Initializes Smartling with the given params
 *
 * @param apiBaseUrl
 * @param projectId
 */
var SmartlingSdk = function (apiBaseUrl, projectId) {
  this.config = {
    apiBaseUrl: apiBaseUrl,
    projectId:  projectId
  };
};

/**
 * Hash of available Smartling operations
 */
SmartlingSdk.OPERATION_URLS = {
  upload: function(projectId) {
    return "/files-api/v2/projects/" + projectId + "/file";
  },
  getOriginal: function(projectId) {
    return "/files-api/v2/projects/" + projectId + "/file";
  },
  status: function(projectId) {
    return "/files-api/v2/projects/" + projectId + "/file/status";
  },
  getTranslations: function(projectId, locale) {
    return "/files-api/v2/projects/" + projectId + "/locales/" + locale + "/file/get-translations";
  },
  filesList: function (projectId) {
    return "/files-api/v2/projects/" + projectId + "/files/list";
  }
};

/**
 * Returns a URL for a Smartling API Operation
 *
 * @param operation         A SmartlingSdk.OPERATIONS value
 * @param smartlingParams   JSON object containing any Smartling parameters
 * @returns {String}
 */
SmartlingSdk.prototype.getSmartlingRequestPath = function(operation, smartlingParams) {
  // ProjectId is always required so provide default settings here
  var params = {};

  _.extend(params, smartlingParams);

  //assemble the request URL
  var requestUrl = this.config.apiBaseUrl + operation;
  requestUrl += '?' + jsonToSearchParameterString(params);

  //console.log('requestUrl', requestUrl);

  return requestUrl;
};

/**
 * Uploads original source content to Smartling (20MB limit for docx and pptx, 10MB limit for all others).
 *
 * https://docs.smartling.com/display/docs/Files+API#FilesAPI-/file/upload(POST)
 *
 * @param fileContents (required)  The file contents to upload.
 * @param fileUri (required)  Value that uniquely identifies the uploaded file. This ID can be used to request the file back.
 *        We recommend you use file path + file name, similar to how version control systems identify the file.
 *        Example: /myproject/i18n/ui.properties.
 * @param fileType (required)
 *        Identifiers: android, ios, gettext, html, javaProperties, yaml, xliff, xml, json, docx, pptx, xlsx, idml
 * @param options (optional)
 * @param options.approved (optional)
 *        This value, either true or false (default), determines whether content in the file is 'approved' (available for translation)
 *        upon submitting the file via the Smartling Dashboard. An error message will return if there are insufficient translation
 *        funds and approved is set to true.
 *        Note: Setting this parameter to true both approves all new content and overrides any locale-specific or global exclusions.
 *        If your workflow includes content exclusions, use this parameter with caution.
 * @param options.smartling.[command] (optional)  Provides custom parser configuration for supported file types. See Supported File Types for more details.
 * @param options.callbackUrl (optional)  A GET request that creates a callback to a URL when a file is 100% published for a locale.
 *        The callback includes these parameters:
 *          fileUri
 *          locale
 *        If you upload another file without a callback URL, it will remove any previous callbackUrl for that file.
 *
 * @return {promise}
 */
SmartlingSdk.prototype.uploadContents = function (fileContents, fileUri, fileType, options) {
  //create a defered object to return
  var deferred = Q.defer();

  //setup default request params
  var smartlingParams = {
    fileUri: fileUri,
    fileType: fileType
  };

  //extend the request params with any options passed in by user
  _.extend(smartlingParams, options);

  //assemble the request URL
  var smartlingUrl = SmartlingSdk.OPERATION_URLS.upload(this.config.projectId);
  var requestUrl = this.getSmartlingRequestPath(smartlingUrl, smartlingParams);

  var req = request.post({
    url: requestUrl
  }, getStandardSmartlingRequestHandler(deferred));

  var form = req.form();
  form.append('file', fileContents, {
    filename: fileUri,
    contentType: 'text/plain'
  });

  //return the promise
  return deferred.promise;
};

SmartlingSdk.prototype.getOriginal = function (fileUri, options) {
  //create a defered object to return
  var defered = Q.defer();

  //setup default request params
  var smartlingParams = {
    fileUri: fileUri
  };

  //extend the request params with any options passed in by user
  _.extend(smartlingParams, options);

  //assemble the request URL
  var smartlingUrl = SmartlingSdk.OPERATION_URLS.getOriginal(this.config.projectId);
  var requestUrl = this.getSmartlingRequestPath(smartlingUrl, smartlingParams);

  var requestParams = {
    url: requestUrl,
    json: true
  };

  //Make the request
  request.get(requestParams, function (error, response, body) {
    if (!error && response.statusCode === 200) {
      defered.resolve(body);
    } else {
      defered.reject(new Error(JSON.stringify(body)));
    }
  });

  //return the promise
  return defered.promise;
};

/**
 * Downloads the requested file (@fileUri) from Smartling.
 *
 * https://docs.smartling.com/display/docs/Files+API#FilesAPI-/file/get(GET)
 *
 * @param fileUri (required)  Value that uniquely identifies the downloaded file.
 *
 * @param fileContents
 * @param locale
 * @param options
 * @param options.locale (optional)  A locale identifier as specified in project setup. If no locale is specified, original content is returned. You can find the list of locales for your project on the Smartling dashboard at https://dashboard.smartling.com/settings/api.
 * @param options.retrievalType (optional)
 *          Allowed values: pending, published, pseudo
 *
 *          pending indicates that Smartling returns any translations (including non-published translations)
 *          published indicates that Smartling returns only published/pre-published translations
 *          pseudo indicates that Smartling returns a modified version of the original text with certain characters transformed and the text expanded. For example, the uploaded string "This is a sample string", will return as "T~hís ~ís á s~ámpl~é str~íñg". Pseudo translations enable you to test how a longer string integrates into your application.
 *          If you do not specify a value, Smartling assumes published.
 * @param options.IncludeOriginalStrings (optional) Allowed values: true, false  For gettext, xml, or json files only.
 *
 * @return {promise}
 */
SmartlingSdk.prototype.getTranslations = function (fileUri, fileContents, locale, options) {
  //create a defered object to return
  var deferred = Q.defer();

  //setup default request params
  var smartlingParams = {
    fileUri: fileUri
  };

  //extend the request params with any options passed in by user
  _.extend(smartlingParams, options);

  //assemble the request URL
  var smartlingUrl = SmartlingSdk.OPERATION_URLS.getTranslations(this.config.projectId, locale);
  var requestUrl = this.getSmartlingRequestPath(smartlingUrl, smartlingParams);
  var requestParams = {
    url: requestUrl,
    json: true
  };

  var req = request.post(
      requestParams,
      function(error, response, body) {
        if (!error && response.statusCode === 200) {
          deferred.resolve(body);
        } else {
          deferred.reject(body);
        }
      }
  );

  var form = req.form();
  form.append('file', fileContents, {
    filename: fileUri,
    contentType: 'text/plain'
  });

  //return the promise
  return deferred.promise;
};

/**
 * Gets status of translations for @fileUri in @locale
 *
 * https://docs.smartling.com/display/docs/Files+API#FilesAPI-/file/status(GET)
 *
 * @param fileUri (required)  Value that uniquely identifies the file.
 *               You can find the list of locales for your project on the Smartling
 *               dashboard at https://dashboard.smartling.com/settings/api.
 *
 * @returns {promise}
 */
SmartlingSdk.prototype.status = function (fileUri) {
  //create a defered object to return
  var deferred = Q.defer();

  //setup default request params
  var smartlingParams = {
    fileUri: fileUri
  };

  //assemble the request URL
  var smartlingUrl = SmartlingSdk.OPERATION_URLS.status(this.config.projectId);
  var requestUrl = this.getSmartlingRequestPath(smartlingUrl, smartlingParams);

  var requestParams = {
    url: requestUrl,
    json: true
  };

  //Make the request
  request.get(requestParams, getStandardSmartlingRequestHandler(deferred));

  //return the promise
  return deferred.promise;
};

SmartlingSdk.prototype.filesList = function (fileUri, fileType) {
  //create a deferred object to return
  var deferred = Q.defer();

  //setup default request params
  var smartlingParams = {
    uriMask: fileUri,
    'fileTypes[]': fileType
  };

  //assemble the request URL
  var smartlingUrl = SmartlingSdk.OPERATION_URLS.filesList(this.config.projectId);
  var requestUrl = this.getSmartlingRequestPath(smartlingUrl, smartlingParams);

  var requestParams = {
    url: requestUrl,
    json: true
  };

  //Make the request
  request.get(requestParams, getStandardSmartlingRequestHandler(deferred));

  //return the promise
  return deferred.promise;
};

//Export the SmartlingSdk Class
module.exports = SmartlingSdk;
