'use strict';

module.exports = function (grunt) {
  var asyncUtil     = require('async'),
      SmartlingTask = require('../lib/smartling-task'),
      _             = require('lodash'),
      chalk         = require('chalk'),
      commonMethods = require('./../lib/common-methods.js');

  var S3_OPTION = "s3Override";

  grunt.registerMultiTask('build-get-translations', 'Gets translations if using on-build translations',
      SmartlingTask.make(function (task, options, sdk, done) {
        commonMethods.setI18nType(grunt, options.i18n);
        var gitBranch = process.env.GIT_BRANCH || 'Unknown';
        var isMasterBranch = (gitBranch === 'origin/master');
        var isCI = process.env.CI;
        var buildUrl = process.env.BUILD_URL || 'Unknown';

        grunt.log.subhead('This might take up to 30 seconds to complete.');

        asyncUtil.waterfall([
          function checkVersion(callback) {
            commonMethods.checkVersion(grunt, options, callback);
          },
          function maybeUploadBase(callback) {
            if (grunt.option(S3_OPTION)) {
              grunt.verbose.writeln('S3 Override is set, skipping upload base');
              // If s3 url option is enabled, Smartling is probably down anyway, so not uploading
              return callback();
            }

            if (!isCI) {
              grunt.verbose.writeln('Not running with CI=true, skipping upload base');
              return callback();
            }

            if (!isMasterBranch) {
              grunt.log.writeln('Not uploading source file because branch is not origin/master');
              grunt.log.writeln('Branch: ' + process.env.GIT_BRANCH);
              return callback();
            }

            commonMethods.uploadSource(sdk, grunt, options, callback);
          },
          function getBuildTranslations(callback) {
            var commandLineOptions = {};

            if (process.env.GIT_COMMIT) {
              commandLineOptions.gitCommit = process.env.GIT_COMMIT;
            }

            if (grunt.option(S3_OPTION)) {
              commandLineOptions['s3-override'] = grunt.option(S3_OPTION);
            } else if (isCI && isMasterBranch) {
              commandLineOptions.saveToS3 = true;
            }

            var retryFunction = commonMethods.getBuildTranslations.bind(this, sdk, grunt, options, commandLineOptions);
            asyncUtil.retry({times: 3, interval: 30000}, retryFunction, function (error, zipFile) {
              if (error) {
                grunt.verbose.writeln('Aborting from getting translations');
                return callback(error);
              }

              grunt.verbose.writeln('Received Zip File Successfully');
              return callback(null, zipFile);
            });
          },
          function unzipTranslations(zipFile, callback) {
            // Check if in CI
            grunt.verbose.writeln('Unzipping Translations');
            commonMethods.unzipTranslations(grunt, zipFile, options, callback);
          },
          function validateTranslations(callback) {
            var i18nType = grunt.option('i18nType');
            try {
              i18nType.validateTranslations();
            } catch (err) {
              return callback(err);
            }
            return callback();
          }
        ], function (error) {
          if (error) {
            // Need to confirm only Errors show up here, but this is to be safe for now
            if (!_.isError(error)) {
              error = grunt.util.error('Get-Translations-Error: ' + error.toString());
            }
            grunt.log.error('There was an error with build-get-translations.', error);

            if (isCI) {
              var qs = {
                type: 'error',
                msg: error.toString(),
                isCI: isCI,
                gitBranch: gitBranch,
                buildUrl: buildUrl
              };

              commonMethods.postStatusToI18nService(grunt, options, qs, function (/* never returns anything */) {
                return done(error);
              });
            } else {
              return done(error);
            }
          } else {
            grunt.log.writeln('Successfully retrieved translations');
            return done();
          }
        });
      })
  );
};
