'use strict';

/*
 * grunt-smartling-sdk
 * https://github.com/hightail/grunt-smartling-sdk
 *
 * Copyright (c) 2014 Hightail
 * Author: Justin Fiedler
 *
 * Licensed under the ISC license.
 */

module.exports = function (grunt) {
  var SmartlingTask = require('../lib/smartling-task'),
      path          = require('path'),
      readlineSync  = require('readline-sync'),
      chalk         = require('chalk'),
      _             = require('lodash'),
      asyncUtil     = require('async'),
      commonMethods   = require('./../lib/common-methods.js');

  function isValidJiraMatch(ticket) {
    var jiraRegex = /[a-zA-Z]{1,10}-\d+/g;
    return ticket.match(jiraRegex) !== null;
  }

  var EXIT_EARLY_ERROR = new Error();
  var COMPARE_MASTER_AND_SMARTLING_FLAG = false;

  grunt.registerMultiTask('submit', 'Upload files to Smartling',
      SmartlingTask.make(function (task, options, sdk, done) {
        var i18nType = commonMethods.setI18nType(grunt, options.i18n);

        var msg = chalk.bold('LOC JIRA Ticket Number (Should have been already created): ');
        var ticket = grunt.option('ticket') || readlineSync.question(msg);

        if (!ticket || !isValidJiraMatch(ticket)) {
          grunt.log.error('Valid JIRA Ticket is required (e.g. LOC-123)');
          return done(grunt.util.error('Bad arguments.'));
        }

        var fileUriForTicket = commonMethods.formatFileName(ticket, options.i18n.fileType);
        var filePathForTicket = path.resolve(fileUriForTicket);

        var failHandler = function (callback, error) {
          if (!_.isError(error)) {
            error = grunt.util.error('Error Submitting to Smartling: ' + JSON.stringify(error, null, 2));
          }

          grunt.log.error('Error Submitting to Smartling', error);

          return callback(error);
        };

        asyncUtil.waterfall([
          function checkVersion(callback) {
            commonMethods.checkVersion(grunt, options, callback);
          },
          function confirmBaseLocaleExists(callback) {
            var baseLocaleFileUri = commonMethods.formatFileName(options.i18n.baseLocale, options.i18n.fileType);
            commonMethods.confirmFileExists(sdk, grunt, options, baseLocaleFileUri, function (error, doesFileExist) {
              if (error) {
                grunt.log.writeln(JSON.stringify(error));
                return callback(error);
              }

              if (doesFileExist) {
                grunt.verbose.writeln('Base File Does exist in Smartling');
                return callback();
              } else {
                grunt.log.error(baseLocaleFileUri + ' does not exist in Smartling, run script to update base file');
                return callback(EXIT_EARLY_ERROR);
              }
            });
          },
          function getNewStrings(callback) {
            // If a file with the ticket name already exists, the user is probably running it a second time
            if (grunt.file.exists(filePathForTicket)) {
              var loadResponse = readlineSync.question(chalk.bold('Detected translation file. Should use it? [y/N] '));
              if (loadResponse.toLowerCase() === 'y') {
                var newStringsFileStr = grunt.file.read(filePathForTicket);
                // TODO: Confirm the file is of valid format
                return callback(null, newStringsFileStr);
              }
            }

            if (grunt.option('localDiff')) {
              grunt.verbose.writeln('Checking local files for new strings');
              var nonBaseLocales = options.i18n.locales.filter(function (locale) {
                return locale !== options.i18n.baseLocale;
              });

              var translations = nonBaseLocales.map(function (locale) {
                var filePath = i18nType.getFilePath({
                  locale: locale,
                  fileName: commonMethods.formatFileName(locale, options.i18n.fileType)
                });

                return i18nType.convertFileStrToObject(grunt.file.read(filePath));
              });
              var baseLocaleTranslations = i18nType.convertFileStrToObject(i18nType.getSourceFileStr());

              // Going through all locale translations and removing anything found in base locale translations
              var result = translations.reduce(function (acc, curr) {
                Object.keys(acc).forEach(function (key) {
                  if (curr.hasOwnProperty(key)) {
                    delete acc[key];
                  }
                });
                return acc;
              }, baseLocaleTranslations);

              return callback(null, i18nType.convertObjectToFileStr(i18nType.removeNoTranslate(result)));
            }

            grunt.verbose.writeln('Checking against Smartling for new files');
            return commonMethods.diffFromSmartlingSource(sdk, grunt, options.i18n, callback);
          },
          function confirmShouldSubmit(newStringsFileStr, callback) {
            grunt.log.subhead('If this is not expected output, you might need to submit differently: https://wiki.twitch.com/display/ENG/I18n+Scripts+Help#I18nScriptsHelp-SubmitStringsManuallymanualsubmit');

            if (newStringsFileStr === null || newStringsFileStr.length === 0) {
              grunt.log.subhead('No new keys were detected.');

              return callback(EXIT_EARLY_ERROR);
            }

            grunt.log.subhead('New Strings To Be Submitted:');
            grunt.log.writeln(newStringsFileStr);

            grunt.verbose.subhead('File To Submit To Smartling:');
            grunt.verbose.writeln(i18nType.convertToSmartlingFormat({fileStr: newStringsFileStr}));

            var msg = chalk.bold('Should submit? (Type N if you want to cancel OR manually edit the list of strings) [y/N] ');
            var submitResponse = grunt.option('confirmSubmit') || readlineSync.question(msg);
            if (submitResponse.toLowerCase() !== 'y') {
              var editMsg = chalk.bold('Do you want to edit the list of strings to submit? [y/N] ');
              var editFileResponse = readlineSync.question(editMsg);
              if (editFileResponse.toLowerCase() !== 'y') {
                return callback(EXIT_EARLY_ERROR);
              }

              if (grunt.file.exists(filePathForTicket)) {
                grunt.log.subhead('File already exists at path: ' + filePathForTicket);
                grunt.log.subhead('Please delete it and run this script again');
                return callback(EXIT_EARLY_ERROR);
              }

              grunt.file.write(filePathForTicket, newStringsFileStr);
              grunt.log.subhead('Saved as: ' + fileUriForTicket);
              grunt.log.subhead('Edit the file and re-run this command and use the same ticket number.');
              return callback(EXIT_EARLY_ERROR);
            }

            return callback(null, newStringsFileStr);
          },

          function preSubmitCheck(newStringsFileStr, callback) {
            if (i18nType.preSubmitCheck) {
              try {
                i18nType.preSubmitCheck({fileStr: newStringsFileStr});
              } catch (err) {
                return callback(err);
              }
            }

            return callback(null, newStringsFileStr);
          },

          function checkStatus(newStringsFileStr, callback) {
            commonMethods.confirmFileExists(sdk, grunt, options, fileUriForTicket, function (error, doesFileExist) {
              if (error) {
                return callback(error);
              }

              if (doesFileExist) {
                var msg = 'File already exists in Smartling.';
                msg += "\nDO NOT SUBMIT AGAIN IF Localization Team has already started processing this ticket, create a new ticket instead.";
                msg += "\nAre you sure you still want to overwrite? [y/N] ";
                var confirmOverwrite = grunt.option('confirmOverwrite') || readlineSync.question(chalk.bold(msg));
                if (confirmOverwrite.toLowerCase() !== 'y') {
                  return callback(EXIT_EARLY_ERROR);
                }
              }

              return callback(null, newStringsFileStr);
            });
          },

          function getSmartlingCallbackUrl(newStringsFileStr, callback) {
            if (grunt.option('skipCallbackUrl')) {
              grunt.verbose.writeln('Skipping Callback Url');
              return callback(null, null, newStringsFileStr);
            }
            commonMethods.getSmartlingCallbackUrl(options, ticket, function (error, callbackUrl) {
              if (error) {
                return callback(error);
              }

              grunt.verbose.writeln('Callback url: ' + callbackUrl);
              return callback(null, callbackUrl, newStringsFileStr);
            });
          },

          function submit(callbackUrl, newStringsFileStr, callback) {
            options.operation.fileType = options.i18n.fileType;
            if (callbackUrl) {
              options.operation.callbackUrl = callbackUrl;
            }
            var smartlingFormatContent = i18nType.convertToSmartlingFormat({fileStr: newStringsFileStr});
            return sdk.uploadContents(smartlingFormatContent, fileUriForTicket, options.i18n.fileType, options.operation)
                .then(function (fileInfo) {
                  if (grunt.option('verbose')) {
                    grunt.log.writeln(JSON.stringify(fileInfo, null, 2));
                  }
                  grunt.log.subhead('Successfully submitted file to Smartling.');
                  return callback(null, newStringsFileStr);
                })
                .fail(failHandler.bind(this, callback));
          },

          function addSubmitJiraComment(newStringsFileStr, callback) {
            if (grunt.option('skipJira')) {
              return callback();
            }

            var formData = {
              ticket: ticket,
              submittedStrings: newStringsFileStr,
              projectId: options.i18n.projectId,
              fileUri: fileUriForTicket
            };

            commonMethods.addSubmitJiraComment(grunt, options, formData, function(error) {
              if (error) {
                grunt.log.error('Could not update JIRA ticket');
                return callback(error);
              }

              grunt.log.subhead('Successfully created comment on JIRA ticket.');

              return callback();
            });
          }
        ], function (error) {
          if (error !== EXIT_EARLY_ERROR) {
            return done(error);
          }

          if (error === EXIT_EARLY_ERROR) {
            grunt.log.error('Did not submit anything');
            if (grunt.option('failOnError')) { // Used for testing
              return done(EXIT_EARLY_ERROR);
            }
          }

          return done();
        });

      })
  );
};