require 'json'
require 'erb'
require 'jira-ruby'
require 'smartling'
require 'lru_redux'
require 'digest'
require './lib/i18n-service/server/v2api'
require 'rest-client'
require 'typhoeus'
require 'aws-sdk'
require './lib/i18n-service/server/utils'
require 'json_web_token'

JIRA_SUBMIT_COMMENT_TEMPLATE = './lib/i18n-service/server/templates/jira-comment-submit.erb'
NUM_LRU_ITEMS = 50
LRU_TIME_OUT_SECS = 2*60*60 # 2 hours
S3_BUCKET_NAME_PREFIX = 'i18n-twitch'

I18N_SMARTLING_CALLBACK_URLS = {
	# 'development' => 'http://requestb.in/1b5ypgf1',
	'development' => 'http://localhost:9293',
	'staging' => 'http://staging-i18n-i18n-smartling-callback-env.uqgmrxaequ.us-west-2.elasticbeanstalk.com',
	'production' => 'http://production-i18n-i18n-smartling-callback.fv28nmfhay.us-west-2.elasticbeanstalk.com'
}

module I18nservice
	class Backend
		def initialize
			# instantiate external clients here, like: @dynamo = Aws::DynamoDB::Client.new(region: "us-west-2")
			@smartling_config = Utils.get_config_file('smartlingConfig')
			creds = Utils.get_config_file('smartlingV2Config');
			@slByAccount = Smartling::Api.new(
				:userId => creds['userId'],
				:userSecret => creds['userSecret']
			)

			if ENV['ENVIRONMENT'] == 'production'
				@s3BucketName = S3_BUCKET_NAME_PREFIX + '-prod'
			else
				@s3BucketName = S3_BUCKET_NAME_PREFIX + '-staging'
			end

			Utils.get_config_file('awsCredentials').each_pair { |k, v| ENV[k] = v}

			if ENV['ENVIRONMENT'] == 'development'
				RestClient.log = 'stdout'
				Typhoeus::Config.verbose = true
				@slByAccount.log = 'stdout'
			end
			
			Aws.config.update({
			  region: 'us-west-2'
			})
			@s3 = Aws::S3::Resource.new
			@bucket = @s3.bucket(@s3BucketName)

			jira_config = Utils.get_config_file('jiraConfig')
			@jira_client = JIRA::Client.new({
				:site         => 'https://jira.twitch.com',
				:context_path => '',
				:consumer_key => 'I18n',
				:private_key_file => Utils.get_config_file_path('jiraPrivKey.pem')
			})

			@jira_client.set_access_token(jira_config['accessToken'], jira_config['accessKey'])

			@jira_comment_submit_template = File.read(JIRA_SUBMIT_COMMENT_TEMPLATE)
			@translationsCache = LruRedux::ThreadSafeCache.new(NUM_LRU_ITEMS, LRU_TIME_OUT_SECS)
			@jwtSecret = Utils.get_config_file('jwtConfig')['secret']
			
			if I18N_SMARTLING_CALLBACK_URLS.has_key? ENV['ENVIRONMENT']
				@i18nSmartlingCallbackUrl = I18N_SMARTLING_CALLBACK_URLS[ENV['ENVIRONMENT']]
			else
				raise 'Invalid environment for Callback Urls'
			end
		end

		def createS3Name(projectId, gitCommit)
			current_time_str = Time.now.getutc.strftime "%Y-%m-%dT%H-%M-%S"
			commit = (gitCommit && gitCommit[0..6]) || 'UnknownCommit'
			return "i18n-service-translations/#{projectId}/#{commit}-#{current_time_str}.zip"
		end

		def s3_put(projectId, gitCommit, zip_str)
			fn = createS3Name(projectId, gitCommit)
			obj = @bucket.put_object(:key => fn, :body => zip_str)
		end

		def s3_get(filename)
			if not @bucket.object(filename).exists?
				return nil
			end

			@bucket.object(filename).get().body.read
		end

		def s3_info
			{
				'prefix' => "https://s3-us-west-2.amazonaws.com/#{@s3BucketName}/",
				'files' => @bucket.objects.map(&:key)
			}
		end

		def s3_op
			# enumerate every object in a bucket
			arr = []
			@bucket.objects.each do |obj|
			  arr.push("#{obj.key} => #{obj.etag}")
			end

			arr.join("\n")
		end

		def s3_op_1
			obj = @bucket.object('i18n-service-translations/')
			puts obj.delete
		end

		def add_project_keys(params)
			if (params.has_key? 'projectId') && (@smartling_config.has_key? params['projectId'])
				newParams = params.clone
				newParams['apiKey'] = @smartling_config[params['projectId']]['apiKey']
				return newParams
			else
				params
			end
		end

		def delete_jira_comment(ticketId, commentId)
			# Unused
			issue = @jira_client.Issue.find(ticketId)

			comment_index = issue.comments.index {|comment| comment.id == commentId}
			raise 'Invalid Comment Id' if comment_index == nil

			issue.comments[comment_index].delete
		end

		def add_jira_comment(ticketId, submittedStrings, projectId, fileUri)
			issue = @jira_client.Issue.find(ticketId)

			if submittedStrings.length > 30000
				submittedStrings = submittedStrings[0..30000] + "\n ... Truncated ..."
			end
			
			bindingModel = binding
			bindingModel.local_variable_set(:submittedStrings, submittedStrings)
			bindingModel.local_variable_set(:projectId, projectId)
			bindingModel.local_variable_set(:ticketId, ticketId)
			bindingModel.local_variable_set(:fileUri, fileUri)

			jira_comment_str = ERB.new(@jira_comment_submit_template).result(bindingModel)
			jira_comment_str = jira_comment_str.force_encoding("UTF-8")

			comment = issue.comments.build
			didSucceed = comment.save!(:body => jira_comment_str)
			raise 'Could not save JIRA comment' if not didSucceed
		end

		def get_smartling_v2_client()
			@slByAccount
		end

		def set_cache(reqHash)
			sha = createSHA(reqHash['file_content'], reqHash['locales'])
			@translationsCache[sha] = {
				'zip_str' => reqHash['zip_str'],
				'updated' => Time.new.to_s,
				'projectId' => reqHash['projectId'],
				'gitCommit' => reqHash['gitCommit'],
				'locales' => reqHash['locales']
			}
			reqHash['zip_str']
		end

		def get_cache(reqHash)
			if reqHash.has_key? 'sha'
				sha = reqHash['sha']
				if @translationsCache.has_key? sha
					return @translationsCache[sha]['zip_str']
				end
			end

			sha = createSHA(reqHash['file_content'], reqHash['locales'])
			if @translationsCache.has_key? sha

				return @translationsCache[sha]['zip_str']
			end

			return nil
		end

		def clear_cache(projectId)
			if projectId == nil
				@translationsCache.clear
			else
				list = @translationsCache.to_a
				list.each { |elem|
					if elem[1]['projectId'] == projectId
						@translationsCache.delete elem[0]
					end
				}
			end
		end

		def cache_info()
			{
				"count" => @translationsCache.count,
				"max_size" => NUM_LRU_ITEMS,
				"ttl" => LRU_TIME_OUT_SECS,
				"valid" => @translationsCache.valid?,
				"currentTime" => Time.new.to_s,
				"cache" => @translationsCache.to_a.collect {|arr| 
					{
						'sha' => arr[0],
						'updated' => arr[1]['updated'],
						'projectId' => arr[1]['projectId'],
						'gitCommit' => arr[1]['gitCommit'],
						'numLocales' => arr[1]['locales'].length,
						'zipLength' => arr[1]['zip_str'].length
					}
				}
			}
		end

		def createSHA(file_content, locales)
			return Digest::SHA256.hexdigest (file_content + locales.to_s)
		end

		def sign_token(ticket, version)
			if version == 'v1'
				key = @jwtSecret
			else
				raise 'Invalid json web token version'
			end

			JsonWebToken.sign({
				ticket: ticket,
				iat: Time.now.to_i
			}, key: key)
		end

		def verify_token(token, version)
			if version != 'v1'
				puts version
				raise 'Invalid version for token verification'
			end
			
			obj = JsonWebToken.verify(token, key: @jwtSecret)
			if obj.has_key? 'error'
				puts obj.error
				raise 'Invalid token for verification'
			end

			obj[:ok]
		end

		def get_i18n_smartling_callback_url()
			@i18nSmartlingCallbackUrl
		end
	end
end
