require 'rest-client'
require 'multi_json'
require 'zip'
require 'typhoeus'
require 'stringio'
require 'rollbar'


SMARTLING_API_URL = 'https://api.smartling.com/v1/'
SMARTLING_API_V2_URL = 'https://api.smartling.com/'
PROXY_REQ_PREFIX = '/smartling-vX-proxy/'
MIN_SCRIPT_VERSION = '1.0.91'
NUM_PARALLEL_REQUESTS = 4
CALLBACK_EXPIRE_TIME = (60*60*24*14) # 2 weeks in seconds

def create_smartling_url(api_url, path)
	return api_url + path[PROXY_REQ_PREFIX.length..-1]
end

module I18nservice
	class Server < TwitchSinatra::Base
		configure do
			mime_type :zip, 'application/zip'
		end

		get '/' do
			Rollbar.info('hi')
			'OK'
		end

		namespace '/api/v1' do
			get '/' do
				# Use the settings.backend object to call operations you define in I18nservice::Backend
			end
		end

		get '/version' do
			# Submit and retrieve might need separate versions
			content_type :json
			{
				'version' => '1.0',
				'minScriptVersion' => MIN_SCRIPT_VERSION
			}.to_json
		end

		get '/smartling-v1-proxy/*' do
			path_for_smartling = create_smartling_url(SMARTLING_API_URL, request.path)
			smartling_params = settings.backend.add_project_keys(params.except('splat', 'captures'))
			RestClient.get(path_for_smartling, params: smartling_params) {
				|response, request, result|
				status response.code
				content_type response.headers[:content_type]
				body response.body
			}
		end

		post '/smartling-v1-proxy/*' do
			path_for_smartling = create_smartling_url(SMARTLING_API_URL, request.path)
			smartling_params = settings.backend.add_project_keys(params.except('splat', 'captures', 'file', 'files'))
			RestClient.post(path_for_smartling, {:file => params[:file][:tempfile], :multipart => true}, params: smartling_params) { 
				|response, request, result| 
				status response.code
				content_type response.headers[:content_type]
				body response.body
			}
		end

		delete '/smartling-v1-proxy/*' do
			path_for_smartling = create_smartling_url(SMARTLING_API_URL, request.path)
			smartling_params = settings.backend.add_project_keys(params.except('splat', 'captures'))
			RestClient.delete(path_for_smartling, params: smartling_params) {
				|response, request, result| 
				status response.code
				content_type response.headers[:content_type]
				body response.body	
			}
		end

		get '/smartling-v2-proxy/*' do
			path_for_smartling = create_smartling_url(SMARTLING_API_V2_URL, request.path)
			split_url = request.path.split('/')
			if split_url.length < 5
				halt 500, 'Invalid Url'
			end

			projectId = split_url[5]
			smartling_params = params.except('splat', 'captures')

			sl = settings.backend.get_smartling_v2_client()

			uri = sl.uri(path_for_smartling, smartling_params)
			results = sl.get_raw(uri)
			
			status results[:response].code
			content_type results[:response].headers[:content_type]
			body results[:response].body
		end

		post '/smartling-v2-proxy/*' do
			path_for_smartling = create_smartling_url(SMARTLING_API_V2_URL, request.path)
			split_url = request.path.split('/')
			if split_url.length < 5
				halt 500, 'Invalid Url'
			end

			projectId = split_url[5]
			
			smartling_params = params.except('splat', 'captures', 'file', 'files')

			sl = settings.backend.get_smartling_v2_client()

			if params[:file] && params[:file].has_key?(:tempfile)
				uri = sl.uri(path_for_smartling)
				results = sl.post_file(uri, smartling_params.merge!({:file => params[:file][:tempfile], :multipart => true}))
			else
				uri = sl.uri(path_for_smartling)
				results = sl.post(uri, smartling_params)
			end
			
			status results[:response].code
			content_type results[:response].headers[:content_type]
			body results[:response].body
		end

		post '/v1/jira/test' do
			settings.backend.add_jira_comment(
				'LOC-229',
				"list of submitted strings",
				'webclient',
				'LOC-229.yaml'
			)
		end


		post '/v1/jira/submit' do
			if (not params.has_key? 'ticket') || (not params['ticket'].start_with? 'LOC')
				halt 400, 'Invalid Ticket'
			end

			if not params.has_key? 'fileUri' || params['fileUri'].empty?
				halt 400, 'No File Uri provided'
			end

			if not params.has_key? 'projectId' || params['projectId'].empty?
				halt 400, 'No Project Id provided'
			end

			if params.has_key? 'submittedStringsEnc'
				submittedStrings = CGI::unescape(params['submittedStringsEnc'])
			elsif params.has_key? 'submittedStrings'
				submittedStrings = params['submittedStrings']
			elsif params.has_key? 'file'
				io = StringIO.new
				file = params[:file][:tempfile]
				file.binmode
				io.write file.read
				submittedStrings = io.string
			else
				halt 400, 'No strings were submitted'
			end

			begin
				settings.backend.add_jira_comment(
					params['ticket'],
					submittedStrings,
					params['projectId'],
					params['fileUri']
				)
			rescue Exception => e
				Rollbar.error(e)
				puts "Error during processing: #{$!}"
				puts "Backtrace:\n\t#{e.backtrace.join("\n\t")}"
				halt 500, e.message
			end
		end


		delete '/v1/jira/comment' do
			# Unused
			if not params['ticket'].start_with? 'LOC'
				halt 400, 'Invalid Ticket'
			end

			if params['comment'].empty?
				halt 400, 'Invalid Comment'
			end

			begin
				settings.backend.delete_jira_comment(
					params['ticket'],
					params['comment']
				)
			rescue Exception => e
				Rollbar.error(e)
				puts "Error during processing: #{$!}"
				puts "Backtrace:\n\t#{e.backtrace.join("\n\t")}"
				halt 500, e.message
			end
		end

		namespace '/api/build/v1' do
			get '/smartling-callback/*/*/*' do
				if (not params.has_key? 'splat') || (not params.has_key? 'fileUri') || (not params.has_key? 'locale')
					puts 'Invalid Arguments'
					Rollbar.error('Invalid Arguments', {:params => params})
					halt 400
				end
				version = params['splat'][0]
				token = params['splat'][1]
				projectId = params['splat'][2]
				fileUri = params['fileUri']
				locale = params['locale']

				parsed_token = settings.backend.verify_token(token, version)

				if parsed_token[:iat] + CALLBACK_EXPIRE_TIME < Time.now.to_i
					halt 200
					Rollbar.error('Smartling-Callback-Expired', :projectId => projectId, :ticket => parsed_token['ticket'])
				end
				if projectId == nil
					puts 'Invalid ProjectId'
					halt 200
				end

				settings.backend.clear_cache(projectId)

				Rollbar.info('Delete Cache', :projectId => projectId, :ticket => parsed_token['ticket'])
				halt 200
			end

			post '/smartling-callback-url' do
				version = params['version'] || 'v1'
				projectId = params['projectId']
				ticket = params['ticket']
				
				if projectId == nil
					halt 500, 'Invalid projectId'
				end

				begin
					token = settings.backend.sign_token(ticket, version)
				rescue Exception => e
					Rollbar.error(e)
					puts "Error during processing: #{$!}"
					puts "Backtrace:\n\t#{e.backtrace.join("\n\t")}"
					halt 500, 'Error creating token: ' + e.message
				end

				"#{settings.backend.get_i18n_smartling_callback_url()}/api-v1/#{version}/#{token}/#{projectId}"
			end

			post '/script-error' do
				type = 'error'
				if ['debug', 'info', 'warning', 'error', 'critical'].include? params['type']
					type = params['type']
				end

				Rollbar.log(type, 'Build-Get-Translations Client-Error', {
					:msg => params['msg'],
					:packageVersion => params['packageVersion'],
					:gitBranch => params['gitBranch'],
					:isCI => params['isCI'],
					:buildUrl => params['buildUrl']
				})
				halt 200
			end

			get '/cache' do
				content_type :json
				body settings.backend.cache_info.to_json
			end

			delete '/cache' do
				if (not params['projectId'].nil?) && params['projectId'].empty?
					halt 400, 'Project Id should be valid'
				end

				if params['projectId']
					settings.backend.clear_cache(params['projectId'])
					Rollbar.info('Delete Cache', :projectId => params['projectId'])
				else
					settings.backend.clear_cache(nil)
					Rollbar.info('Delete Cache')
				end

				content_type :json
				body settings.backend.cache_info.to_json
			end

			get '/s3' do
				content_type :json
				body_data = settings.backend.s3_info()
				body body_data.to_json
			end

			get '/translations-cache' do
				if not params.has_key? 'sha' or params['sha'].empty?
					halt 400, 'SHA required'
				end
				
				zip_str = settings.backend.get_cache({
					'sha' => params['sha']
				})
				if zip_str == nil
					halt 404
				end

				content_type :zip
				body zip_str
			end

			# needed params: projectId, locales, fileUri, file
			post '/get-translations' do
				projectId = params['projectId']
				gitCommit = params['gitCommit']
				locales = (params['locales'] || '').split(',').map {|locale| locale.strip }
				fileType = params['fileType'] || 'yaml'
				
				halt 400, 'Locales should not be empty' if locales.empty?

				if (params.has_key? 's3-override') && (not params['s3-override'].empty?)
					# if s3 override is enabled, return it immediately. Another endpoint?
					s3_obj = settings.backend.s3_get(params['s3-override'])
					
					if s3_obj == nil
						halt 404, 'Invalid S3 File'
					end

					Rollbar.info('Build-Get-Translations S3-Override')
					content_type :zip
					return s3_obj
				end

				io = StringIO.new
				file = params[:file][:tempfile]
				file.binmode
				io.write file.read
				file_content = io.string

				cache_val = settings.backend.get_cache({'file_content' => file_content, 'locales' => locales})
				if not cache_val.nil?
					Rollbar.info('Build-Get-Translations Used-Cache')
					puts 'Cache hit!'
					binary_data = cache_val
				else
					sl = settings.backend.get_smartling_v2_client()

					hydra = Typhoeus::Hydra.new(max_concurrency: NUM_PARALLEL_REQUESTS)
					requestObjects = locales.map do |locale|
						fileName = locale + '.' + fileType
						temp_file = Tempfile.new(fileName)
						temp_file.write file_content
						temp_file.rewind

						path_for_smartling = SMARTLING_API_V2_URL + "files-api/v2/projects/#{projectId}/locales/#{locale}/file/get-translations"
						params_with_file = {
							:file => temp_file,
							'fileUri' => params['fileUri'],
							'retrievalType' => 'published',
							'includeOriginalStrings' => false
						}

						begin
							token_header = sl.token_header()
						rescue Exception => e
							Rollbar.error(e)
							puts "Error during processing: #{$!}"
							puts "Backtrace:\n\t#{e.backtrace.join("\n\t")}"
							halt 500, e.message
						end

						request = Typhoeus::Request.new(
							path_for_smartling,
							method: :post,
							body: params_with_file,
							headers: {
								Authorization: token_header
							}
						)
						
						hydra.queue(request)
						{:request => request, :fileName => fileName}
					end
					hydra.run

					requestObjects.each do |requestObject|
						if requestObject[:request].response.code != 200
							Rollbar.error('Build-Get-Translations Bad-Smartling-Response', {
								:fileName => requestObject[:fileName],
								:status => requestObject[:request].response.code,
								:body => requestObject[:request].response.body
							})
							halt 500, "Get-Translations from Smartling Failed: #{requestObject[:fileName]}\n" + requestObject[:request].response.body
						end
					end

					zip_stream = Zip::OutputStream.write_buffer do |zio|
						requestObjects.each do |requestObject|
							next if requestObject[:fileName] == params[:file][:filename]
							
							zio.put_next_entry(requestObject[:fileName])
							zio.write requestObject[:request].response.body
						end
						# Creating this after because Smartling returns an empty file for the base locale
						zio.put_next_entry(params[:file][:filename])
						zio.write file_content
					end

					binary_data = zip_stream.string
					settings.backend.set_cache({
						'file_content' => io.string,
						'zip_str' => binary_data,
						'locales' => locales,
						'projectId' => projectId,
						'gitCommit' => gitCommit
					})
					
				end

				if params.has_key? 'saveToS3' # if should save to S3, do it here
					settings.backend.s3_put(projectId, gitCommit, binary_data)
				end

				Rollbar.info('Build-Get-Translations', :gitCommit => gitCommit)
				content_type :zip
				body binary_data
			end
		end

	end
end
