require 'rest-client'
require 'multi_json'
require 'smartling/uri'

# Future note: Initially when I started this service, I used many of Smartling's API functions and thought
# using the sdk would be helpful. However, over time, it ended up only being used for it's ability to get tokens.
# It was necessary to do some monkey patching for custom features, but if I had the chance to go back and start again,
# I would have not used the SDK to begin with.
# Tech Debt: Remove Smartling-sdk and implement simple oauth/token code
module Smartling
  class Api
    # Monkey patching Smartling's call method because it does processing before returning the results, I'd like the raw results
    def call(uri, method, auth, upload, download, params = nil)
      headers = {}
      headers[:Authorization] = token_header() if auth
      headers[:content_type] = :json unless upload
      headers[:accept] = :json unless download
      RestClient::Request.execute(:method => method, 
                                  :url => uri.to_s,
                                  :payload => params,
                                  :headers => headers,
                                  :timeout => 120)  {|response, request, result|
        return {:response => response, :request => request, :result => result}
      }
    end

    # Smartling swallows errors and returns a non-descriptive error, but the errors are good to have
    def format_api_error(res)
      STDERR.puts "\e[31m#{res}\e[0m"
      raise (res.to_s)
    end

    # Authenticate - /auth-api/v2/authenticate (POST)
    def token()
      # Check if current token is still valid
      if @token
        now = Time.new.to_i
        if @token_expiration - 180 > now # Changing to get token 60 seconds before expiration
          return @token
        elsif @refresh && @refresh_expiration > now
          return refresh()
        end
      end

      # Otherwise call authenticate endpoint
      uri = uri('auth-api/v2/authenticate', {}, {})
      RestClient.post(uri.to_s, {:userIdentifier => @userId, :userSecret => @userSecret}.to_json, {:content_type => :json, :accept => :json}) {|res, _, _|
        process_auth(process(res))
        return @token
      }
    end

    # Overriding because of a typo in the original
    # Refresh Authentication - /auth-api/v2/authenticate/refresh (POST)
    def refresh()
      uri = uri('auth-api/v2/authenticate/refresh', {}, {})
      RestClient.post(uri.to_s, {:refreshToken => @refresh}.to_json, {:content_type => :json, :accept => :json}) {|res, _, _|
        process_auth(process(res))
        return @token
      }
    end
  end
end
