const _ = require('lodash');
const intlParser = require('intl-messageformat-parser');
const {sprintf} = require("sprintf-js");

const {isObject} = require('./utils.js');
const {validPluralKeys} = require('./constants');

const notSupportedTypes = {
  selectFormat: true
};

/**
 * @param {Object} value
 * @throws {Error} If not valid ICU Message Format
 */
function icuLinter(value) {
  if (_.isString(value)) {
    throw new Error(sprintf('value should not be a string: %s', value));
  }

  if (!value.phrase) {
    throw new Error(sprintf('Invalid Value Object: %s', JSON.stringify(value, null, 2)));
  }
  let ast = null;
  try {
    ast = intlParser.parse(value.phrase);
  } catch (err) {
    err.message = 'Error parsing ICU Msg Fmt string: ' + JSON.stringify(value.phrase) + '. Error: ' + err.message;
    throw err;
  }

  if (!ast) {
    throw new Error(sprintf("Invalid ICU Message from string: %s", value.phrase));
  }

  lintPlurals(ast, value.phrase);

  checkUsingSupportedFeatures(ast);
}

function lintPlurals(ast, phrase) {

  let numPlurals = _.reduce(ast.elements, function reducer(acc, curr) {
    if (curr.type === "argumentElement" && curr.format && curr.format.type === "pluralFormat") {
      return acc + 1;
    }

    if (_.isArray(curr.elements)) {
      return acc + _.reduce(obj.elements, reducer);
    }

    return acc;
  }, 0);

  if (numPlurals > 1) {
    throw new Error(sprintf("Invalid Number of Plurals in string: %s", phrase));
  }

  if (numPlurals === 1) {
    if (ast.elements.length !== 1) {
      throw new Error(sprintf("Plurals must encapsulate whole string: %s", phrase));
    }
  }
}

/**
 * TODO: Make sure not using nested general select statements
 * @param {Object} obj
 */
function checkUsingSupportedFeatures(obj) {
  if (!isObject(obj)) {
    return;
  }

  if (_.get(obj, "format.type") in notSupportedTypes) {
    throw new Error(sprintf('%s is not currently supported.', obj.type));
  }

  if (_.get(obj, "format.ordinal")) {
    throw new Error(sprintf("Ordinal statements is not supported. For id: %s", obj.id));
  }

  let offset = _.get(obj, "format.offset");
  if (offset !== undefined && offset !== 0) {
    throw new Error(sprintf("Offsets are not supported for: %s", obj.id));
  }

  let options = _.get(obj, "format.options");
  if (Array.isArray(options)) {
    options.forEach((option) => {
      if (option.selector.startsWith("=")) {
        throw new Error(sprintf("=Values are not supported: %s", obj.id));
      }
    });
  }

  if (_.isArray(obj.elements)) {
    obj.elements.forEach(checkUsingSupportedFeatures);
  }
}

function checkNotUsingReservedKeywords(key) {
  if (key in validPluralKeys) {
    throw new Error(`Should not use the plural categories as a key name: ${key}`);
  }
}

/**
 * Checks if object has valid keys
 * @param {Object.<string, Object>} obj
 * @throws {Error} If there is some sort of invalid state
 */

function runLints(obj) {
  _.mapKeys(obj, function (value, key) {

    // Lint Key
    checkNotUsingReservedKeywords(key);

    // Lint Value
    icuLinter(value);
  });
}

function getParams(str) {
  let ast = intlParser.parse(str);

  return Object.keys(getParamsHelper(ast));
}

function getParamsHelper(ast) {
  return ast.elements.reduce(function (params, el) {
    let {format, id, type} = el;

    if (type === 'messageTextElement') {
      return params;
    }

    if (!format) {
      params[id] = true;
      return params;
    }

    let formatType = format.type.replace(/Format$/, '');

    switch (formatType) {
      case 'number':
      case 'date':
      case 'time':
        params[id] = true;
        return params;
      case 'plural':
      case 'selectOrdinal':
      case 'select':
        params[id] = true;
        return format.options.reduce(function (obj, option) {
          return Object.assign({}, obj, getParamsHelper(option.value));
        }, params);
    }
  }, {})
}

module.exports = {
  runLints,
  icuLinter,
  checkNotUsingReservedKeywords,
  lintPlurals,
  getParams
};
