const jsyaml = require('js-yaml');
const fs = require('fs');
const _ = require('lodash');

function loadYaml(str) {
  return jsyaml.safeLoad(str);
}

function dumpYaml(yamlObj) {
  if (Object.keys(yamlObj).length === 0) {
    return '';
  }

  return jsyaml.dump(yamlObj);
}

function loadYamlFile(filePath) {
  return jsyaml.safeLoad(getFileContent(filePath));
}

function dumpYamlFile(filePath, yamlObj) {
  fs.writeFileSync(filePath, jsyaml.dump(yamlObj));
}

function fileExists(filePath) {
  return fs.existsSync(filePath);
}

/**
 * @param obj
 * @returns {string}
 */
function printYamlWithNullsRemoved(obj) {
  return removeNulls(jsyaml.dump(obj));
}

/**
 * Is used to remove nulls from a yaml string
 * @param {string} yamlString
 * @returns {void|string|*|XML}
 */
function removeNulls(yamlString) {
  /* Because the YAML file format is deterministic, we can be extra picky here */
  // Replacing a null, \t: to account for new lines in different OSes, and a new line
  return yamlString.replace(/: null[ \t]*\n/g, ':\n');
}

function getFileContent(filePath) {
  return fs.readFileSync(filePath, 'utf8');
}

function removeNoTranslate(sourceObj) {
  return _.pickBy(sourceObj, (val) => {
    return !(isObject(val) && val.noTranslate);
  });
}

/**
 * @param {object} obj
 * @returns {boolean}
 */
function isObject(obj) {
  return !_.isNull(obj) && _.isPlainObject(obj);
}

module.exports = {
  isObject,
  fileExists,
  removeNoTranslate,
  loadYaml,
  loadYamlFile,
  dumpYaml,
  dumpYamlFile,
  getFileContent,
  printYamlWithNullsRemoved
};
