const _ = require('lodash');

const { validPluralKeys } = require('./constants.js');

const validPluralKeysArray = _.keys(validPluralKeys);
const intlParser = require('intl-messageformat-parser');


/**
 * Remember SmartlingFormat uses KeyStrings and plural/one/other
 * We want just {phrase: "ICU_STRING"} OR {} if there isn't anything there
 * @param obj
 * @returns {Object}
 */
function smartlingToPropSchema(obj) {
  let pluralKeys = obj.pluralKeys || {};
  delete obj.pluralKeys;
  let transformedObj = _.mapValues(obj, function (val, key) {
    if (_.isString(val)) {
      return {
        phrase: val
      };
    }

    let icuValueOrNull = convertToICU(val, pluralKeys[key] || val.plural);
    if (icuValueOrNull) {
      return {
        phrase: icuValueOrNull
      };
    }

    return null;
  });

  // If null is returned, that means a plural phrase did not have any translations. So removing them
  Object.keys(transformedObj).forEach((key) => {
    if (transformedObj[key] === null) {
      delete transformedObj[key];
    }
  });

  return transformedObj;
}

function insertPhraseProperty(yamlObj) {
  return _.mapValues(yamlObj, (val) => {
    return {
      phrase: val
    };
  });
} 

/**
 * Used by translation-reducer in ember-intl-i18n-twitch and in grunt submit
 * Extracts the phrase out of an object, doing this because in the future it's good to know all
 * the parts of the code that need to be changed if this format changes.
 * @param yamlObj
 * @returns {Object}
 */
function extractPhrase(yamlObj) {
  return _.mapValues(yamlObj, (val) => {
    if (val && val.phrase) {
      return val.phrase;
    }
    return null;
  });
}

/**
 * Converts an object into ICU String
 * Object should only have keys from {@link validPluralKeys}
 * @param {object} obj
 * @param pluralKey
 * @returns {string}
 */
function convertToICU(obj, pluralKey) {
  if (!pluralKey) {
    throw new Error('No Valid Plural Key for: ' + JSON.stringify(obj));
  }
  let filteredObject = _.pick(obj, validPluralKeysArray);
  if (Object.keys(filteredObject).length === 0) {
    return null;
  }

  let cldrSubs = _.toPairs(filteredObject).map(function (pair) {
    return `${pair[0]} {${pair[1]}}`;
  });

  return `{${pluralKey}, plural, ${cldrSubs.join(' ')}}`;
}

/**
 * Complete for our use case, taken from: https://github.com/yahoo/intl-messageformat-parser/issues/10
 * @param {Object} ast
 * @returns {string}
 */
function printICUMessage(ast) {
  return ast.elements.reduce(function (message, el) {
    let {format, id, type, value} = el;

    if (type === 'messageTextElement') {
      return message + value;
    }

    if (!format) {
      return `${message}{${id}}`;
    }

    let formatType = format.type.replace(/Format$/, '');

    let style, offset, options, optionValue;
    switch (formatType) {
      case 'number':
      case 'date':
      case 'time':
        style = format.style ? `, ${format.style}` : '';
        return `${message}{${id}, ${formatType}${style}}`;

      case 'plural':
      case 'selectOrdinal':
      case 'select':
        offset = format.offset ? `, offset:${format.offset}` : '';
        options = format.options.reduce(function (str, option) {
          optionValue = printICUMessage(option.value);
          return `${str} ${option.selector} {${optionValue}}`;
        }, '');
        return `${message}{${id}, ${formatType}${offset},${options}}`;
    }
  }, '');
}

/*
 * Returns plural key/values like {one: '', other: ''} or null from string
 */
function getICUPluralParts(str) {
  if (!str) {
    return null;
  }

  let ast = intlParser.parse(str);

  if (ast.elements && ast.elements[0]) {
    let elem = ast.elements[0];

    if (elem.type === 'argumentElement' && elem.format && elem.format.type === 'pluralFormat') {
      let res = {};
      res.plural = elem.id;

      elem.format.options.forEach(function (option) {
        res[option.selector] = printICUMessage(option.value);
      });

      return res;
    }
  }

  return null;
}

module.exports = {
  extractPhrase,
  convertToICU,
  printICUMessage,
  getICUPluralParts,
  smartlingToPropSchema,
  insertPhraseProperty
};
