const { assert } = require('chai');
const { describe, it } = require('mocha');

const { runLints, lintPlurals, getParams } = require('./../src/linters.js');
const { loadYamlFile } = require('./../src/utils.js');


//noinspection JSCheckFunctionSignatures
describe('Test ICU Object Linter', function () {
  it('Should pass valid ICU Objects', function () {  
    const icuObj = {
      header: {
        phrase: 'header'
      },
      footer: {
        phrase: 'footer'
      }
    };

    assert.doesNotThrow(function () { runLints(icuObj); });
  });

  it('Should fail invalid ICU Value', function () {
    const icuObj = {
      header: {
        phrase: '{aNum, plural'
      },
      footer: {
        phrase: 'footer'
      }
    };

  assert.throws(function () {runLints(icuObj);}, /SyntaxError/);
  });

  it('Should fail when using reserved key names incorrectly', function () {
    assert.throws(function () {runLints({one: 'one'});},  Error, 'Should not use the plural categories');
    assert.throws(function () {runLints({many: 'one'});},  Error, 'Should not use the plural categories');
  });

  it('Should pass valid ICU strings file', function () {
    assert.doesNotThrow(function () {runLints(loadYamlFile('./yaml/strings.yaml'));});
  });

  it('Should fail if using select statement', function () {
    let obj = {
      hello: {
        phrase: '{aVar, select, one {One statement} other {other statement}}'
      }
		};

    assert.throws(function () { runLints(obj); }, Error, 'is not currently supported');
  });

  it('Should fail if using selectordinal statement', function () {
    let obj = {
      hello: {
        phrase: '{aVar, selectordinal, one {One statement} other {other statement}}'
      }
    };

    assert.throws(function () { runLints(obj); }, Error, 'Ordinal statements is not supported');
  });

  it('Should fail if using =Value in statement', function () {
    let obj = {
      hello: {
        phrase: '{aVar, plural, one {One statement} other {other statement} =10 {when there are 10}}'
      }
    };

    assert.throws(function () { runLints(obj); }, Error, '=Values are not supported');
  });

  it('Should expect plural format to be correct', function () {
    let obj = {
      hello: {
        plural: 'numViewers',
        one: '1 viewer'
      }
    };

    assert.throws(function () { runLints(obj); }, Error, 'Invalid Value Object');
  });

  it('Verify plurals are root if they exist and only one exist', function () {
    let obj = {
      'key': {
        phrase: 'This should {aNum, plural, one {Not be} other {supported}}'
      }
    };

    assert.throws(function () { runLints(obj); }, Error, 'Plurals must encapsulate');

    obj = {
      'key': {
        phrase: 'This {aNum, plural, one {Not be} other {supported}} should {aNum, plural, one {Not be} other {supported}}'
      }
    };

    assert.throws(function () { runLints(obj); }, Error, 'Invalid Number of Plurals');

    obj = {
      'key': {
        phrase: 'This should {aNum, plural, one {{anotherNum, plural, one {} other {}} Not be} other {supported}}'
      }
    };

    assert.throws(function () { runLints(obj); }, Error);

    obj = {
      'key': {
        phrase: 'This is fine'
      }
    };

    assert.doesNotThrow(function () { runLints(obj); });

    obj = {
      key: {
        phrase: '{dayCount,number} {dayCount, plural, one {day} other {days} }'
      }
    };

    assert.throws(function () { runLints(obj); }, Error);
  });
});

describe('Test Params Extractor', function () {
  it('Should extract params', function () {
    assert.deepEqual(getParams('Sorry, we couldn\'t find anyone named "{user}"'), ['user']);

    assert.deepEqual(getParams('hi {name}'), ['name']);

    assert.deepEqual(getParams('hi {name} the date is {date}'), ['name', 'date']);

    assert.deepEqual(getParams('hi {name} the date is {date}. Number of the day is {num, number}'), ['name', 'date', 'num']);

    assert.deepEqual(getParams('{viewerCount, plural, one {1 viewer} other {# viewers}}'), ['viewerCount']);

    assert.deepEqual(getParams('{viewerCount, plural, one {1 viewer} other {# viewers on {channel}}}'), ['viewerCount', 'channel']);

    assert.deepEqual(getParams('hi nothing here {viewerCount, plural, one {1 viewer} other {# viewers}}'), ['viewerCount']);

  });
});

