const { assert } = require('chai');
const { describe, it } = require('mocha');

const { diffObjects, mergeObjects, setValuesToNull, getKeysInCommon } = require('./../src/object-operations.js');

describe('setValuesToNull', function () {
  it('Should work for simple object', function () {
    assert.deepEqual(setValuesToNull({a:1}), {a: null});
  });
});

describe('getKeysInCommon', function () {
  it('Should return first object if rest are empty', function () {
    assert.deepEqual(getKeysInCommon([{a:1}]), {a: 1});
  });

  it('Should work for empty secondary objects', function () {
    const testObjects = [{a: {b: {c: null}, e: 1}, d: 'aString'}, {}, {}];

    assert.deepEqual(getKeysInCommon(testObjects), {});
  });

  it('Should work for empty primary objects', function () {
    const testObjects = [{}, {a: {b: {c: null}, e: 1}, d: 'aString'}, {}, {}];

    assert.deepEqual(getKeysInCommon(testObjects), {});
  });

});

describe('Diff Objects', function () {
  it('Should give back first object for invalid inputs', function () {
    assert.deepEqual(diffObjects({}, null), {});
    assert.deepEqual(diffObjects(null, {}), null);
    assert.deepEqual(diffObjects(null, null), null);
    assert.deepEqual(diffObjects([], []), []);
  });

  it('Should work for empty objects', function () {
    assert.deepEqual(diffObjects({},{}), {});
  });

  it('Should work for empty second object', function () {
    assert.deepEqual(diffObjects({a:1, b: {c: 'aString'}},{}), {a:1, b: {c: 'aString'}});
  });

  it('Should work for complex objects', function () {
    assert.deepEqual(
        diffObjects(
            {a: 1, b: {c: 'aString'}, d: {e: false}, f: {}},
            {a: 0, d: {e: true}, f: {}}),
        {b: {c: 'aString'}});
  });

  it('Returns keys in first that are not in second', function () {
    let arr = diffObjects({
      'first': null,
      'second': null,
      'fourth': null
    },{
      'second': null,
      'third': null
    });

    assert.deepEqual(arr, { first: null, fourth: null});
  });
});

describe('Merge Objects', function () {
  it('Should merge empty objects', function () {
    assert.deepEqual(mergeObjects({}, {}, {}), {});
  });

  it('Should merge in order of objects', function () {
    assert.deepEqual(mergeObjects({a: 1, d: 0}, {b: 1}, {d: 1}), {a: null, d:1});
  });

  it('Should English merge correctly', function () {
    let sourceFileObj = {
      hello: 'hello',
      info_numViewers: '{numViewers, plural, one {1 viewer} other {{numViewers} viewers}}',
      alreadyExists: 'already exists'
    };
    let translationFileObj = {
      hello: 'not-en hello',
      info_numViewers: '{numViewers, plural, other {not-en {numViewers} viewers}}',
      alreadyExists: 'not-en already exists'
    };
    let newTranslationsFileObj = {
      info_numViewers: '{numViewers, plural, many {not-en {numViewers} viewers}}'
    };

    let result = mergeObjects(sourceFileObj, translationFileObj, newTranslationsFileObj);

    let expectedResult = {
      hello: 'not-en hello',
      info_numViewers: '{numViewers, plural, many {not-en {numViewers} viewers}}',
      alreadyExists: 'not-en already exists'
    };

    assert.deepEqual(result, expectedResult);
  });
});
