const { assert } = require('chai');
const { describe, it } = require('mocha');

const printYaml = require('./../src/print-smartling-format.js');

describe('Print Yaml', function () {
  it('Print Yaml without comments or content', function () {
    assert.deepEqual(printYaml(undefined, undefined), '');
    assert.deepEqual(printYaml({}, {}), '');
    assert.deepEqual(printYaml({a: {}}), '');
    assert.deepEqual(printYaml({a: {phrase: '1'}}), 'a: \'1\'');
  });

  it('Print Yaml with comments', function () {  
    assert.deepEqual(printYaml({a: {
      comment: 'a comment',
      phrase: '1'
    }}), '# a comment\na: \'1\'');
  });

  it('Print Yaml with Quote', function () {
    let expected = `# a comment
a: This is what I'll do`;
    assert.deepEqual(printYaml({a: {
      comment: 'a comment',
      phrase: "This is what I'll do"
    }}), expected);
  });

  it('Print Yaml with plural phrase', function () {
    let expected =
`a:
  # a comment
  one: 1 view
  other: '{numView} views'
# smartling.sltrans = notranslate
pluralKeys:
  a: numViews
# smartling.sltrans = translate`;

    assert.deepEqual(printYaml({a: {
      comment: 'a comment',
      phrase: '{numViews, plural, one {1 view} other {{numView} views}}'
    }}), expected);
  });

  it('Print Yaml with plural phrase with single quotes', function () {
    let expected =
`a:
  # a comment
  one: 1 viewer's opinion
  other: '{numView} viewer''s opinions'
# smartling.sltrans = notranslate
pluralKeys:
  a: numViews
# smartling.sltrans = translate`;

    assert.deepEqual(printYaml({a: {
      comment: 'a comment',
      phrase: '{numViews, plural, one {1 viewer\'s opinion} other {{numView} viewer\'s opinions}}'
    }}), expected);
  });

  it('Print Yaml with long strings', function () {
    let expected =
`a:
  # a comment
  one: >-
    12 ads ad asdasdasdadasd asd a ds as d as d as d asdasdasdasdasda dasdad
    asdasdasdjlkasdfjoiasfnoaisudfhoiasnfiaushdfasdf 
  other: >-
    2 ads ad asdasdasdadasd asd a ds as d as d as d asdasdasdasdasda dasdad
    asdasdasdjlkasdfjoiasfnoaisudfhoiasnfiaushdfasdf 
# smartling.sltrans = notranslate
pluralKeys:
  a: numViews
# smartling.sltrans = translate`;

    assert.deepEqual(printYaml({a: {
      comment: 'a comment',
      phrase: "{numViews, plural, one {12 ads ad asdasdasdadasd asd a ds as d as d as d asdasdasdasdasda dasdad asdasdasdjlkasdfjoiasfnoaisudfhoiasnfiaushdfasdf } other {2 ads ad asdasdasdadasd asd a ds as d as d as d asdasdasdasdasda dasdad asdasdasdjlkasdfjoiasfnoaisudfhoiasnfiaushdfasdf }}"
    }}), expected);
  });
});

