const {assert} = require('chai');
const {describe, it} = require('mocha');

const intlMessageParser = require('intl-messageformat-parser');

const {printICUMessage, convertToICU} = require('./../src/yaml-convert.js');

describe('Printing ICU Message', function () {
  function convertToASTAndAssertEquals(str) {
    const ast = intlMessageParser.parse(str);
    assert.strictEqual(printICUMessage(ast), str);
  }

  it('Should print simple messages correctly', function () {
    convertToASTAndAssertEquals('Hello World!');
    convertToASTAndAssertEquals('{mood} of viewers');
  });

  it('Should print messages with selects correctly', function () {
    convertToASTAndAssertEquals('Started on: {aDate, date, short}');
    convertToASTAndAssertEquals('Started on: {aTime, time, short}');
    convertToASTAndAssertEquals('{numViewers, plural, one {1 viewer} other {{numViewers} viewers}}');
    convertToASTAndAssertEquals('I have {numCats, number} cats.');
    // offsets don't work fully
    // convertToASTAndAssertEquals('{numViewers, plural, offset:1 one {1 viewer} other {{numViewers} viewers}}');

  });
});

describe('Converting Plural ICU Object', function () {
  it('Should convert plural object', function () {
    assert.strictEqual(convertToICU({
      one: '1 viewer',
      other: '# viewers'
    }, 'numViewers'), '{numViewers, plural, one {1 viewer} other {# viewers}}');
  });

  it('Should throw Exception on invalid plural key', function () {
    assert.throws(function () {
      convertToICU({
        one: '1 viewer',
        other: '# viewers'
      }, '');
    }, Error, 'No Valid Plural Key');
  });
});
