var path = require('path');
var fs = require('fs');

module.exports = function(grunt) {

  grunt.loadNpmTasks('grunt-contrib-concat');
  grunt.loadNpmTasks('grunt-contrib-uglify');
  grunt.loadNpmTasks('grunt-contrib-clean');

  const supportedLocales = generateLocales(grunt);

  // Generates { destPath: sourcePath } for the concat and uglify (minify) targets
  const localeTargets = generateLocaleTargets(supportedLocales);

  grunt.initConfig({
    pkg: grunt.file.readJSON('package.json'),
    locales: supportedLocales,
    outputFolder: 'dist',
    localesFolder: 'locale-data',

    clean: {
      dist: {
        src: ['dist']
      },
      nonessentials: ['dist/concated', 'dist/locale-data', 'dist/Intl.min.js']
    },
    concat: {
      locales: {
        options: {
          banner: '(function(undefined) {',
          footer: "\n}).call('object' === typeof window && window || 'object' === typeof self && self || 'object' === typeof global && global || {});"
        },
        files: localeTargets.concat
      }
    },
    uglify: {
      locales: {
        files: localeTargets.uglify
      }
    }
  });

  grunt.registerTask('dist', ['clean', 'extract', 'concat:locales', 'uglify:locales'/*, 'clean:nonessentials' */]);

  grunt.registerTask('extract', function () {
    const LocalesPath = path.dirname(require.resolve('intl/locale-data/jsonp/en.js'));
    const IntlMinPath = require.resolve('intl/dist/Intl.min.js');
    const distPath = grunt.config('outputFolder');

    grunt.file.copy(IntlMinPath, path.join(distPath, 'Intl.min.js'));

    fs.readdirSync(LocalesPath).forEach(function (file) {
      const locale = file.slice(0, file.indexOf('.'));
      if (grunt.config('locales').indexOf(locale) >= 0) {
        grunt.file.copy(path.join(LocalesPath, file), path.join(distPath, grunt.config('localesFolder'), file));
        console.log('Moved locale file: ', file);
      }
    });
  });
};

function generateLocales(grunt) {
  // Locales.json (currently) has the fully qualified ones. 
  // This also gets the two character language locales as well in case it's needed in the future
  const locales = grunt.file.readJSON('locales.json').locales;
  var obj = {};
  locales.forEach((locale) => {
    if (locale.indexOf('-') >= 0) {
      const language = locale.substring(0, locale.indexOf('-'));
      obj[language] = true;
    }
    obj[locale] = true;
  });

  return Object.keys(obj);
}

function generateLocaleTargets(locales) {
    const concatObj = {};
    const uglifyObj = {};
    
    locales.forEach((locale) => {
      concatObj['dist/concated/' + locale + '.js'] = ['dist/Intl.min.js', path.join('dist', 'locale-data', locale + '.js')];
      uglifyObj['dist/min/' + locale + '.js'] = ['dist/concated/' + locale + '.js'];
    });
    return {concat: concatObj, uglify: uglifyObj};
  }