#!/usr/bin/env python2.7
import sys
import os
import subprocess

import boto.utils

USAGE="./{} vol_group_name logical_vol_name mount_point"
STRIPE_SIZE="2M"
FS_TOOL="mkfs.xfs"

def is_ebs_vol(ref):
    return ref.startswith("ebs")


def get_block_devices():
    return boto.utils.get_instance_metadata()['block-device-mapping']


def get_ebs_block_devices(device_map):
    return [d for r, d in device_map.items() if is_ebs_vol(r)]


def qualify_name(d):
    """
    Returns a fully qualified path to the block device, or None if one cannot be
    found

    http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-attaching-volume.html

    Asgard requests our devices be mapped under /dev/sdX, however the driver
    is permitted to remap those as it sees fit, which mean they can end up in the
    /dev/xvdX space
    """
    if not d:
        return None

    # check for presence in /dev/sdX
    qualified = '/dev/{}'.format(d)
    if os.path.exists(qualified):
        return qualified

    # check for presences in /dev/xvdX
    qualified = '/dev/xvd{}'.format(d[-1])
    if os.path.exists(qualified):
        return qualified

    return None


def get_qualified_names(devices):
    return map(qualify_name, devices)


def _call(*args):
    return subprocess.check_call(args)


def pv_create(devices):
    _call("pvcreate", *devices)


def vg_create(vg_name, devices):
    _call("vgcreate", vg_name, *devices)


def lv_create(lv_name, vg_name, devices):
    args = ["-i", str(len(devices)), # number of stripes
            "-I", STRIPE_SIZE,       # stripe size
            "-l", "100%FREE",        # logical volume size
            "-n", lv_name,           # logical volume name
            vg_name]                 # volume group to use

    _call("lvcreate", *args)


def mkfs(vg_name, lv_name):
    lvm_path = "/dev/mapper/{}-{}".format(vg_name, lv_name)
    _call(FS_TOOL, lvm_path)
    return lvm_path


def mount(lvm_name, mount_point):
    _call("mount", lvm_path, mount_point)


if __name__ == '__main__':
    if len(sys.argv)-1 != 3:
        sys.exit(USAGE.format(sys.argv[0]))

    vg_name = sys.argv[1]
    lv_name = sys.argv[2]
    mount_point = sys.argv[3]

    devices = get_ebs_block_devices(get_block_devices())
    devices = get_qualified_names(devices)
    pv_create(devices)
    vg_create(vg_name, devices)
    lv_create(lv_name, vg_name, devices)
    lvm_path = mkfs(vg_name, lv_name)
    mount(lvm_path, mount_point)
