package main

import (
	"code.google.com/p/gopacket"
	"code.google.com/p/gopacket/layers"
	"encoding/json"
	"flag"
	"fmt"
	"io"
	"log"
	"math/rand"
	"net"
	"os"
	"runtime"
	"strings"
	"sync"
	"time"
)

var iface = flag.String("interface", "en0", "interface to scan")
var workers = flag.Int("workers", 2, "number of workers to spin up")
var sample = flag.Float64("sample", 1.0, "sample rate if any")
var tcp = flag.String("tcp", "", "tcp addr to ship json output to")
var logstats = flag.Bool("logstats", false, "causes the program to log pcap stats to stdout")

var internalCount, externalCount int64

type DNSResponse struct {
	IP  net.IP
	DNS *layers.DNS
}

var TwitchNetStrings = []string{
	"10.0.0.0/8",
}

var TwitchNets []*net.IPNet

func init() {
	mustParseTwitchNets()
}

func mustParseTwitchNets() {
	for _, netString := range TwitchNetStrings {
		_, ipNet, err := net.ParseCIDR(netString)
		if err != nil {
			log.Fatal(err)
		}

		TwitchNets = append(TwitchNets, ipNet)
	}
}

func parseDNS(buf []byte, linkType layers.LinkType) (*DNSResponse, bool) {
	packet := gopacket.NewPacket(buf, linkType, gopacket.DecodeOptions{Lazy: true, NoCopy: true})
	appLayer := packet.ApplicationLayer()
	if appLayer != nil && !layers.LayerTypeDNS.Contains(appLayer.LayerType()) {
		return nil, false
	}

	ip4Layer := packet.Layer(layers.LayerTypeIPv4)
	ip4, ok := ip4Layer.(*layers.IPv4)
	if !ok {
		return nil, false
	}

	dnsLayer := packet.Layer(layers.LayerTypeDNS)
	dns, ok := dnsLayer.(*layers.DNS)
	if !ok {
		return nil, false
	}

	return &DNSResponse{
		IP:  ip4.DstIP,
		DNS: dns,
	}, true
}

func packetFilter(out chan<- *DNSResponse, in <-chan []byte, linkType layers.LinkType, wg *sync.WaitGroup) {
	defer wg.Done()
	for buf := range in {
		resp, ok := parseDNS(buf, linkType)
		if !ok {
			continue
		}

		hasExternal := false
		for _, q := range resp.DNS.Questions {
			name := string(q.Name)

			if strings.HasSuffix(name, "justin.tv") || strings.HasSuffix(name, "twitch.tv") {
				continue
			}

			if strings.HasSuffix(name, ".in-addr.arpa") {
				continue
			}

			hasExternal = true
		}

		if hasExternal {
			out <- resp
		}
	}
}

func txtPacketLogger(w io.Writer, in <-chan *DNSResponse, wg *sync.WaitGroup) {
	defer wg.Done()
	for resp := range in {
		var info []string

		info = append(info, fmt.Sprintf("dst=%s", resp.IP))

		for i, q := range resp.DNS.Questions {
			info = append(info, fmt.Sprintf("q[%d]=%+q", i, q.Name))
		}

		for _, a := range resp.DNS.Answers {
			info = append(info, fmt.Sprintf("a[%+q]=%s", a.Name, a.IP))
		}

		_, err := fmt.Fprintf(w, "%s\n", strings.Join(info, " "))
		if err != nil {
			log.Printf("failed to write txt log line: %s", err)
			return
		}
	}
}

type DnsAnswer struct {
	Name string `json:"name"`
	Ip   string `json:"ip"`
}

type DnsPacket struct {
	Ip        string      `json:"ip"`
	Questions []string    `json:"questions"`
	Answers   []DnsAnswer `json:"answers"`
}

func jsonPacketLogger(w io.Writer, in <-chan *DNSResponse, wg *sync.WaitGroup) {
	defer wg.Done()

	encoder := json.NewEncoder(w)

	for resp := range in {
		packet := DnsPacket{}
		packet.Ip = resp.IP.String()

		for _, q := range resp.DNS.Questions {
			packet.Questions = append(packet.Questions, string(q.Name))
		}

		for _, a := range resp.DNS.Answers {
			answer := DnsAnswer{
				Name: string(a.Name),
				Ip:   a.IP.String(),
			}

			packet.Answers = append(packet.Answers, answer)
		}

		if err := encoder.Encode(packet); err != nil {
			log.Printf("failed to encode json: %s", err)
			return
		}
	}
}

func sampleResponses(out chan<- *DNSResponse, in <-chan *DNSResponse, sampleRate float64) {
	defer close(out)
	for resp := range in {
		if rand.Float64() < sampleRate {
			out <- resp
		}
	}
}

func main() {
	log.SetFlags(0)
	flag.Parse()

	var writers []io.Writer
	writers = make([]io.Writer, *workers)

	for i := 0; i < *workers; i++ {
		if *tcp != "" {
			conn, err := net.Dial("tcp", *tcp)
			if err != nil {
				log.Fatal(err)
			}
			writers[i] = conn
		} else {

			writers[i] = os.Stdout
		}
	}

	if *workers < 2 {
		log.Fatalf("minimum number of workers is 2")
	}

	runtime.GOMAXPROCS(*workers)

	// big buffered channel for spikes
	inPackets := make(chan []byte, 1024*1024)
	filteredPackets := make(chan *DNSResponse, 1024*1024)
	sampledPackets := make(chan *DNSResponse, 1024*1024)

	udpCap, err := NewUdpCap(inPackets)
	if err != nil {
		log.Fatal(err)
	}

	if *logstats {
		go func() {
			for _ = range time.Tick(1 * time.Second) {
				udpCap.LogStats()
			}
		}()
	}

	filterWg := &sync.WaitGroup{}
	loggerWg := &sync.WaitGroup{}

	filterWg.Add(*workers)
	for i := 0; i < *workers; i++ {
		go packetFilter(filteredPackets, inPackets, udpCap.LinkType(), filterWg)
	}

	if *sample < 1.0 {
		go sampleResponses(sampledPackets, filteredPackets, *sample)
	} else {
		sampledPackets = filteredPackets
	}

	loggerWg.Add(*workers)
	for i := 0; i < *workers; i++ {
		go jsonPacketLogger(writers[i], sampledPackets, loggerWg)
	}

	exit := make(chan error, 3)

	go func() {
		exit <- fmt.Errorf("pcap exit: %s", udpCap.Run())
	}()

	go func() {
		filterWg.Wait()
		exit <- fmt.Errorf("filter worker died")
	}()

	go func() {
		loggerWg.Wait()
		exit <- fmt.Errorf("logging worker died")
	}()

	log.Fatal(<-exit)
}
