package main

import (
	"code.google.com/p/gopacket/layers"
	"code.google.com/p/gopacket/pcap"
	"log"
	"runtime"
	"sync/atomic"
)

type UdpCap struct {
	logStats int64
	stop     int64
	out      chan<- []byte
	started  chan struct{}
	handle   *pcap.Handle
}

func NewUdpCap(ch chan<- []byte) (*UdpCap, error) {
	handle, err := pcap.OpenLive(*iface, 512, false, pcap.BlockForever)
	if err != nil {
		return nil, err
	}

	if err := handle.SetBPFFilter("udp and port 53"); err != nil {
		return nil, err
	}

	if runtime.GOOS != "darwin" {
		if err := handle.SetDirection(pcap.DirectionOut); err != nil {
			return nil, err
		}
	}

	return &UdpCap{
		handle:  handle,
		out:     ch,
		started: make(chan struct{}),
	}, nil
}

func (c *UdpCap) LogStats() {
	atomic.AddInt64(&c.logStats, 1)
}

func (c *UdpCap) Stop() {
	atomic.AddInt64(&c.stop, 1)
}

func (c *UdpCap) LinkType() layers.LinkType {
	return c.handle.LinkType()
}

func (c *UdpCap) Run() error {
	close(c.started)
	defer c.handle.Close()

	for atomic.LoadInt64(&c.stop) == 0 {
		buf, _, err := c.handle.ZeroCopyReadPacketData()
		if err != nil {
			return err
		}

		c.out <- buf

		if atomic.LoadInt64(&c.logStats) > 0 {
			atomic.AddInt64(&c.logStats, -1)

			stats, err := c.handle.Stats()
			if err != nil {
				return err
			} else {
				log.Printf("received=%d dropped=%d ifdropped=%d", stats.PacketsReceived, stats.PacketsDropped, stats.PacketsIfDropped)
			}
		}
	}

	return nil
}
