package main

import (
	"encoding/json"
	"flag"
	"fmt"
	elastigo "github.com/mattbaird/elastigo/lib"
	"io"
	"log"
	"os"
	"runtime"
	"sync"
	"sync/atomic"
	"time"
)

var from = flag.String("from", "", "index to pull data from")
var to = flag.String("to", "", "index to push data to")
var create = flag.Bool("create", false, "create the 'to' index if not exist")
var host = flag.String("host", "", "elasticsearch host")
var size = flag.Int("size", 5000, "scroll size")
var workers = flag.Int("workers", 4, "number of bulk writers")

func checkFlags() {
	if *from == "" {
		log.Fatal("from must not be empty")
	}

	if *to == "" {
		log.Fatal("to must not be empty")
	}

	if *host == "" {
		log.Fatal("host must not be empty")
	}
}

func retryFunc(fn func() (interface{}, error), timeout time.Duration) (interface{}, error) {
	deadline := time.Now().Add(timeout)
	delay := 10 * time.Millisecond

	var resp interface{}
	var err error
	for {
		resp, err = fn()
		if err != nil {
			log.Printf("retryFunc err: %s, retrying in %s", err, delay)
		} else {
			return resp, err
		}

		time.Sleep(delay)
		delay *= 2

		if time.Now().After(deadline) {
			return nil, err
		}
	}
}

// {
//     "_id": "7B6xwWhgQYWg87ha4R9bLA",
//     "_index": "logstash-2015.04.01",
//     "_score": 0.0,
//     "_source": {
//         "@timestamp": "2015-04-01T00:00:53.636Z",
//         "@version": "1",
//         "byte_count": 1500,
//         "dest_ip": "90.223.216.228",
//         "dest_port": 65123,
//         "host": "199.9.253.201:32567",
//         "packet_count": 1,
//         "protocol": "TCP",
//         "sflow_host": "netmon4.sfo01.justin.tv",
//         "source_ip": "199.9.255.178",
//         "source_port": 80,
//         "type": "sflow"
//     },
//     "_type": "sflow"
// },

func main() {
	log.SetFlags(0)
	log.SetOutput(os.Stdout)
	flag.Parse()
	checkFlags()

	runtime.GOMAXPROCS(runtime.NumCPU())

	c := elastigo.NewConn()
	c.Domain = *host

	if ok, err := c.ExistsIndex(*from, "", nil); err != nil {
		log.Fatalf("error checking if index %s exists: %s", *from, err)
	} else if !ok {
		log.Fatalf("no such index: %s", *from)
	}

	if ok, err := c.ExistsIndex(*to, "", nil); err != nil && err != elastigo.RecordNotFound {
		log.Fatalf("error checking if index %s exists: %s", *to, err)
	} else if !ok && !*create {
		log.Fatalf("no such index: %s", *to)
	} else if ok && *create {
		log.Fatalf("index already exists: %s", *to)
	}

	if *create {
		if _, err := c.CreateIndex(*to); err != nil {
			log.Fatal(err)
		}
	}

	resp, err := c.Search(*from, "", map[string]interface{}{
		"search_type": "scan",
		"scroll":      "1m",
	}, map[string]interface{}{
		"query": map[string]interface{}{
			"match_all": map[string]interface{}{},
		},
		"size": *size,
	})
	if err != nil {
		log.Fatal(err)
	}

	scrollId := resp.ScrollId

	var sofar int64
	var total int64

	go func() {
		start := time.Now()
		for _ = range time.Tick(5 * time.Second) {
			done := atomic.LoadInt64(&sofar)
			tot := atomic.LoadInt64(&total)

			rate := float64(done) / time.Now().Sub(start).Seconds()
			eta := time.Now().Add(time.Second * time.Duration(float64(tot-done)/rate))

			log.Printf("indexed %d of %d (%.02f%%), rate=%d/sec eta=%s", done, tot, 100.0*float64(done)/float64(tot), int64(rate), eta)
		}
	}()

	ch := make(chan elastigo.Hits)
	wg := &sync.WaitGroup{}

	wg.Add(*workers)
	for i := 0; i < *workers; i++ {
		go func() {
			defer wg.Done()
			for hits := range ch {
				pr, pw := io.Pipe()
				encoderDone := make(chan error, 1)

				go func(w io.WriteCloser, hits elastigo.Hits, done chan error) {
					defer w.Close()
					encoder := json.NewEncoder(w)
					for _, hit := range hits.Hits {
						action := map[string]interface{}{
							"index": map[string]interface{}{
								"_id":   hit.Id,
								"_type": hit.Type,
							},
						}

						if err := encoder.Encode(action); err != nil {
							done <- err
							return
						}

						if _, err := w.Write(*hit.Source); err != nil {
							done <- err
							return
						}

						if _, err := w.Write([]byte{'\n'}); err != nil {
							done <- err
							return
						}
						atomic.AddInt64(&sofar, 1)
					}
				}(pw, hits, encoderDone)

				if _, err := c.DoCommand("POST", fmt.Sprintf("/%s/_bulk", *to), nil, pr); err != nil {
					log.Fatal(err)
				}
			}
		}()
	}

	defer wg.Wait()
	for {
		log.Printf("scroll request")
		raw, err := retryFunc(func() (interface{}, error) {
			return c.Scroll(map[string]interface{}{"scroll": "1m"}, scrollId)
		}, 60*time.Second)
		if err != nil {
			log.Fatalf("scroll error: %s", err)
		}

		hits := raw.(elastigo.SearchResult).Hits

		if len(hits.Hits) == 0 {
			log.Printf("scroll ended")
			return
		} else {
			log.Printf("processing %d hits", len(hits.Hits))
		}

		atomic.StoreInt64(&total, int64(hits.Total))

		ch <- hits
	}
}
