package varnishcollector

import (
	"bufio"
	"bytes"
	"eventfilter/event"
	"io"
	"log"
	"os/exec"
	"strings"
)

// Collects varnish logs using varnishncsa

type VarnishCollector struct {
	eventChannel chan event.Event
	count        int64
	lastCount    int64
	isProd       bool
	fqdn         string
}

func NewVarnishCollector(eventChannel chan event.Event, isProd bool) *VarnishCollector {
	out, err := exec.Command("/bin/hostname", "-f").Output()
	if err != nil {
		log.Fatal(err)
	}
	vc := VarnishCollector{
		eventChannel,
		0,
		0,
		isProd,
		strings.TrimSpace(string(out)),
	}
	return &vc
}

func (vc *VarnishCollector) parseNCSALine(line string) event.Event {
	e := event.NewEvent()

	fields := strings.Split(line, " ")

	e.Data["host"] = vc.fqdn
	e.Data["time"] = fields[0]
	e.Data["elapsed"] = fields[1]
	e.Data["status"] = fields[2]
	e.Data["hit"] = fields[3]
	e.Data["method"] = fields[4]
	e.Data["url"] = fields[5]
	e.Data["proto"] = fields[6]
	e.Data["real-ip"] = strings.Split(fields[7], ",")[0]
	e.Data["ip"] = fields[8]
	e.Data["client-id"] = fields[9]

	return e
}

func (vc *VarnishCollector) parseNCSALines(reader io.Reader) {
	scanner := bufio.NewScanner(reader)
	for scanner.Scan() {
		event := vc.parseNCSALine(scanner.Text())
		vc.count++
		vc.eventChannel <- event
	}
}

func (vc *VarnishCollector) Run() {
	args := make([]string, 0, 0)
	if vc.isProd {
		args = []string{
			"-n",
			"/dev/shm/varnish",
			"-F",
			"%{%FT%TZ%z}t %D %s %{Varnish:hitmiss}x %r %{X-Forwarded-For}i %h %{Client-ID}i",
		}
	} else {
		args = []string{
			"-F",
			"%{%FT%TZ%z}t %D %s %{Varnish:hitmiss}x %r %{X-Forwarded-For}i %h %{Client-ID}i",
		}
	}
	cmd := exec.Command("/usr/bin/varnishncsa", args...)
	var outerr bytes.Buffer
	cmd.Stderr = &outerr

	outpipe, err := cmd.StdoutPipe()
	if err != nil {
		log.Fatal(err)
	}

	err = cmd.Start()
	if err != nil {
		log.Fatal(err)
	}
	vc.parseNCSALines(outpipe)
}

func (vc *VarnishCollector) DumpStatus() {
	log.Println("VarnishCollector Collected: ", vc.count)
	log.Println("VarnishCollector Collected in interval:", vc.count-vc.lastCount)
	vc.lastCount = vc.count
}
