//
// Analysis of postgres query data
//

package main

import (
	"encoding/json"
	"flag"
	"log"
	"net/http"
	"os"
	"time"

	"eventfilter/collector/varnishcollector"
	"eventfilter/event"
	"eventfilter/filter/varnishappfilter"
	"eventfilter/output/logstashoutput"
	"runtime"
)

type Configuration struct {
	NumProcs   int    "Number of processors to use"
	Pprof      bool   "Run the pprof server"
	Address    string "Logstash address to report these events to"
	IsProd     bool   "Run varnish collector in production mode"
	IsFiltered bool   "Run in filtered mode"
	IsUDP      bool   "Run in UDP mode (defaults to true)"
}

func NewConfiguration() Configuration {
	return Configuration{
		4,
		false,
		"localhost:9000",
		false,
		true,
		true,
	}
}

var (
	config Configuration
)

func init() {
	// FIXME: I don't like the way that configfiles and command line options interact
	// Come up with a better solution.
	configfileName := flag.String("configfile", "", "JSON configuration file")
	pprof := flag.Bool("pprof", false, "run pprof server (true, false)")
	logstashAddress := flag.String("address", "", "Logstash server address")
	isProd := flag.Bool("isprod", false, "connect to varnish as prod (/dev/shm/varnish)")
	isFiltered := flag.Bool("isfiltered", true, "filter output (defaults to true)")
	isUDP := flag.Bool("isudp", true, "use logstash UDP protocol")

	// Parse all our configuration.
	flag.Parse()

	config = NewConfiguration()
	if *configfileName != "" {
		cf, err := os.Open(*configfileName)
		if err != nil {
			log.Fatal(err)
		}
		decoder := json.NewDecoder(cf)
		err = decoder.Decode(&config)
		if err != nil {
			log.Fatal(err)
		}
		cf.Close()
	}

	if *logstashAddress != "" {
		config.Address = *logstashAddress
	}

	if *pprof {
		config.Pprof = true
	}

	if *isProd {
		config.IsProd = true
	}

	config.IsFiltered = *isFiltered
	config.IsUDP = *isUDP
}

func main() {
	// runtime.GOMAXPROCS(config.NumProcs)
	runtime.GOMAXPROCS(4)

	if config.Pprof {
		go func() {
			log.Println(http.ListenAndServe("0.0.0.0:6060", nil))
		}()
	}

	// go PrintStats()

	// log.Println(config)

	eventQueue := make(chan event.Event, 1000)
	outputQueue := make(chan event.Event, 1000)

	// Initialize collector
	collector := varnishcollector.NewVarnishCollector(eventQueue, config.IsProd)

	// Initialize filters
	filter := varnishappfilter.NewVarnishAppFilter(eventQueue, config.IsFiltered)

	// Initialize outputs
	// output := printoutput.NewPrintOutput(output_queue)
	// output := syslogoutput.NewSyslogOutput(output_queue, config.Name)
	output := logstashoutput.NewLogstashOutput(outputQueue, config.Address, config.IsUDP)

	// Run all the parts of the chain (from tail to front)
	go output.Run()
	go filter.Run(outputQueue)
	go collector.Run()

	// Status output
	ticker := time.NewTicker(5 * time.Second)
	for {
		select {
		case <-ticker.C:
			collector.DumpStatus()
			filter.DumpStatus()
			output.DumpStatus()
		}
	}
}
