package querystats

// Given a stream of queries, collects aggregate statistics about them.

import (
	"fmt"
	"pgtools/pgquery"
	"pgtools/querycollector"
	"sort"
	"sync/atomic"
	"time"
)

var (
	query_stats map[string]*PGQueryStat
	first_time  time.Time
	last_time   time.Time
)

type PGQueryStat struct {
	QueryNormalized string
	Count           int64
	ElapsedUsec     int64
}

func NewPGQueryStat(n string) *PGQueryStat {
	return &PGQueryStat{
		QueryNormalized: n,
	}
}

func (s *PGQueryStat) addQuery(q pgquery.PGQuery) {
	atomic.AddInt64(&s.Count, 1)
	atomic.AddInt64(&s.ElapsedUsec, int64(q.GetElapsedTimeSeconds()*1000000000))
}

type PGQueryStatSlice []*PGQueryStat

func (s PGQueryStatSlice) Len() int {
	return len(s)
}

func (s PGQueryStatSlice) Swap(i, j int) {
	s[i], s[j] = s[j], s[i]
}

func (s PGQueryStatSlice) Less(i, j int) bool {
	return s[i].ElapsedUsec < s[j].ElapsedUsec
}

func AddQueryStat(q pgquery.PGQuery) {
	n := q.GetQueryStringNormalized()
	stats := query_stats[n]
	if stats == nil {
		stats = NewPGQueryStat(n)
		query_stats[n] = stats
	}
	stats.addQuery(q)
}

func init() {
	query_stats = make(map[string]*PGQueryStat)
}

func Dump() {
	// Sort and print output
	s := make(PGQueryStatSlice, 0, len(query_stats))
	for _, d := range query_stats {
		s = append(s, d)
	}

	sort.Sort(s)
	for _, d := range s {
		fmt.Printf("%5.4f %9d : %s\n", float64(d.ElapsedUsec)/1000000000.0, d.Count, d.QueryNormalized)
	}
	fmt.Println(last_time)
}

func CollectQueries(qc querycollector.QueryCollector) {
	query_channel := qc.GetQueryChannel()

	// Pull one off first to initialize
	query := <-query_channel
	first_time = query.GetTime()
	AddQueryStat(query)

	for {
		query := <-query_channel
		AddQueryStat(query)
		last_time = query.GetTime()
	}
}
