package main

import (
	"container/list"
	"fmt"
)

type ByteSize int

const (
	B ByteSize = 1 << (iota * 10)
	KB
	MB
	GB
	TB
	PB
)

// this buffer continues to flush when in closes
func Buffer(in <-chan []byte, maxSize ByteSize, statsTag string) <-chan []byte {
	out := make(chan []byte)

	stats := NewStatsLogger(fmt.Sprintf("buffer.%s", statsTag))
	stats.Add("drop", 0)

	go func() {
		defer stats.Stop()
		defer close(out)

		size := 0
		ls := list.New()
		inOpen := true

		for {
			if ls.Len() == 0 {
				// read the first byte, or exit after
				// flushing the last byte
				buf, ok := <-in
				if !ok {
					return
				}

				ls.PushBack(buf)
				size += len(buf)
			} else if inOpen {
				// continue to read/write as long as in is open
				front := ls.Front()
				select {
				case buf, ok := <-in:
					if !ok {
						inOpen = false
						continue
					}

					if size+len(buf) > int(maxSize) {
						stats.Add("drop", 1)
					} else {
						ls.PushBack(buf)
						size += len(buf)
					}
				case out <- front.Value.([]byte):
					ls.Remove(front)
					size -= len(front.Value.([]byte))
				}
			} else {
				// keep flushing bytes until we're empty
				front := ls.Front()
				out <- front.Value.([]byte)
				ls.Remove(front)
				size -= len(front.Value.([]byte))
			}
		}
	}()

	return out
}
