package main

import (
	"bufio"
	"encoding/json"
	"fmt"
	"golang.org/x/net/context"
	"log"
	"net"
	"sync"
	"time"
)

func Listen(ctx context.Context, ln net.Listener, tag string) <-chan []byte {
	wg := &sync.WaitGroup{}
	out := make(chan []byte)
	done := make(chan struct{})

	stats := NewStatsLogger(fmt.Sprintf("listener.%s", tag))

	// use a select in case the listener closes early
	go func() {
		select {
		case <-ctx.Done():
		case <-done:
		}
		ln.Close()
	}()

	// accept and run handlers
	go func() {
		defer close(out)
		defer wg.Wait()
		defer close(done)
		defer stats.Stop()

		for {
			conn, err := ln.Accept()
			if err != nil {
				log.Printf("listener on %s exit: %s", ln.Addr(), err)
				return
			}

			wg.Add(1)
			go doHandle(ctx, out, conn, tag, stats, wg)
		}
	}()

	return out
}

func doHandle(ctx context.Context, out chan<- []byte, conn net.Conn, tag string, stats StatsLogger, wg *sync.WaitGroup) {
	defer wg.Done()
	ctx, cancel := context.WithCancel(ctx)
	defer cancel()

	// ensure connection gets closed
	go func() {
		<-ctx.Done()
		conn.Close()
	}()

	scanner := bufio.NewScanner(TimeoutConn(conn, 5*time.Second))

	for scanner.Scan() {
		var obj map[string]interface{}

		if err := json.Unmarshal(scanner.Bytes(), &obj); err != nil {
			stats.Add("drop", 1)
			continue
		}

		if _, ok := obj["type"].(string); !ok || tag != "" {
			obj["type"] = tag
		}

		obj["@timestamp"] = time.Now().In(time.UTC).Format(time.RFC3339)
		obj["@version"] = 1

		encoded, err := json.Marshal(obj)
		if err != nil {
			stats.Add("drop", 1)
			continue
		}

		out <- encoded

		stats.Add("ship", 1)
	}

	log.Printf("%s - connection ended with: %s", conn.RemoteAddr(), scanner.Err())
}
