package main

import (
	"flag"
	"fmt"
	"github.com/crowdmob/goamz/aws"
	"github.com/crowdmob/goamz/s3"
	"golang.org/x/net/context"
	"log"
	"net"
	"net/http"
	_ "net/http/pprof"
	"os"
	"os/signal"
	"runtime"
	"strconv"
	"sync"
	"syscall"
	"time"
)

func dupMessages(in <-chan []byte, outs ...chan<- []byte) {
	for msg := range in {
		for _, out := range outs {
			out <- msg
		}
	}
}

func mustGetBucket(bucketName string) *s3.Bucket {
	auth, err := aws.GetAuth("", "", "", time.Time{})
	if err != nil {
		log.Fatalf("failed to get aws auth: %s", err)
	}

	s3conn := s3.New(auth, aws.USWest2)
	bucket := s3conn.Bucket(bucketName)

	return bucket
}

func main() {
	log.SetFlags(0)
	flag.Parse()

	runtime.GOMAXPROCS(runtime.NumCPU())

	go func() {
		log.Println(http.ListenAndServe(":6060", nil))
	}()

	cfg := mustParseConfig()

	bucket := mustGetBucket(cfg.S3.Bucket)

	// all messages that go to logstash will
	// get coalesced into this channel
	lsch := make(chan []byte)
	lsbuf := Buffer(lsch, 512*MB, "logstash")

	// all s3 logs will get coalesced here
	alls3logs := make(chan *S3Log)

	// boot up listeners
	for _, listener := range cfg.Listeners {
		ln, err := net.Listen("tcp", net.JoinHostPort("", strconv.Itoa(listener.Port)))
		if err != nil {
			log.Fatal(err)
		}

		// inbound messages come on this channel
		lnch := Listen(context.Background(), ln, listener.Type)

		// outbound s3 messages get pushed here
		s3ch := make(chan []byte)
		s3logs := PackForS3(Buffer(s3ch, 256*MB, fmt.Sprintf("s3.%s", listener.Type)), listener.Type)

		// dup all messages to both s3 and logstash
		go func() {
			for msg := range lnch {
				s3ch <- msg
				lsch <- msg
			}
		}()

		// coalesce packed s3 logs into one channel
		go func() {
			for {
				alls3logs <- <-s3logs
			}
		}()
	}

	var wg sync.WaitGroup
	// boot up logstash writers
	for i := 0; i < cfg.Logstash.Workers; i++ {
		wg.Add(1)
		go func() {
			WriteToLogstash(cfg.Logstash.Addr, lsbuf)
			wg.Done()
		}()
	}

	for i := 0; i < cfg.S3.Workers; i++ {
		wg.Add(1)
		go func() {
			WriteToS3(bucket, alls3logs)
			wg.Done()
		}()
	}

	sig := make(chan os.Signal, 1)
	signal.Notify(sig, syscall.SIGHUP)

	// wait for sighup/sigint
	<-sig
}
