package main

import (
	"bytes"
	"fmt"
	"log"
	"os"
	"time"
)

var s3hostname string

func init() {
	mustGetHostname()
}

func mustGetHostname() {
	h, err := os.Hostname()
	if err != nil {
		log.Fatal(err)
	}

	s3hostname = h
}

type S3Log struct {
	Key   string
	Parts []*bytes.Buffer
}

var (
	PART_SIZE  = 1024 * 1024 * 10
	PART_COUNT = 10
)

func genKey(tag string) string {
	now := time.Now().In(time.UTC)
	return fmt.Sprintf("%d/%s.%s.%s.txt", now.Second(), tag, s3hostname, now.Format(time.RFC3339))
}

func NewS3Log(tag string) *S3Log {
	l := &S3Log{
		Key: genKey(tag),
	}

	if _, ok := l.newPart(); !ok {
		log.Fatalf("this is a bug, failed to create a new s3 log buffer")
	}

	return l
}

func (l *S3Log) newPart() (*bytes.Buffer, bool) {
	if len(l.Parts) >= PART_COUNT {
		return nil, false
	}

	part := bytes.NewBuffer(make([]byte, 0, PART_SIZE))
	l.Parts = append(l.Parts, part)

	return part, true
}

func (l *S3Log) Append(line []byte) bool {
	part := l.Parts[len(l.Parts)-1]

	if part.Len()+len(line)+1 > PART_SIZE {
		var ok bool
		part, ok = l.newPart()
		if !ok {
			return false
		}
	}

	// make sure we actually have the last part
	part = l.Parts[len(l.Parts)-1]

	if _, err := part.Write(line); err != nil {
		log.Fatalf("failed to append byte buffer: %s", err)
	}

	if _, err := part.Write([]byte{'\n'}); err != nil {
		log.Fatalf("failed to append byte buffer: %s", err)
	}

	return true
}

func PackForS3(in <-chan []byte, tag string) chan *S3Log {
	out := make(chan *S3Log)

	go func() {
		defer close(out)

		s3log := NewS3Log(tag)
		for msg := range in {
			if ok := s3log.Append(msg); !ok {
				out <- s3log
				s3log = NewS3Log(tag)
			} else {
				s3log.Append(msg)
			}
		}
	}()

	return out
}
