package main

import (
	"bytes"
	"github.com/crowdmob/goamz/s3"
	"log"
	"time"
)

var s3writerStats = NewStatsLogger("s3writer")

func WriteToS3(bucket *s3.Bucket, in <-chan *S3Log) {
	for logfile := range in {
		shipLog(bucket, logfile)
	}
}

func shipLog(bucket *s3.Bucket, logfile *S3Log) {
	multi := retryRequest(func() (interface{}, error) {
		return bucket.InitMulti(logfile.Key, "text/plain", s3.Private, s3.Options{})
	}).(*s3.Multi)

	completedParts := make(chan s3.Part, len(logfile.Parts))
	for i, p := range logfile.Parts {
		go func(n int, raw *bytes.Buffer) {
			completedParts <- retryRequest(func() (interface{}, error) {
				return multi.PutPart(n, bytes.NewReader(raw.Bytes()))
			}).(s3.Part)
		}(i+1, p)
	}

	var parts []s3.Part
	for i := 0; i < len(logfile.Parts); i++ {
		part := <-completedParts
		s3writerStats.Add("bytes", part.Size)
		parts = append(parts, part)
	}

	retryRequest(func() (interface{}, error) {
		return nil, multi.Complete(parts)
	})
	log.Printf("s3writer shipped logfile: %s", logfile.Key)
	s3writerStats.Add("ship", 1)
}

// func NewS3Writer(bucketName string, in <-chan []byte, logTag string) (*S3Writer, error) {
// 	hostname, err := os.Hostname()
// 	if err != nil {
// 		return nil, err
// 	}

// 	auth, err := aws.GetAuth("", "", "", time.Time{})
// 	if err != nil {
// 		return nil, err
// 	}

// 	awsConnection := s3.New(auth, aws.USWest2)
// 	bucket := awsConnection.Bucket(bucketName)

// 	return &S3Writer{
// 		bucket: bucket,
// 		in:     in,
// 		prefix: fmt.Sprintf("all.s3.%s", hostname),
// 	}, nil
// }

func retryRequest(f func() (interface{}, error)) interface{} {
	delay := 50 * time.Millisecond
	for {
		resp, err := f()
		if err == nil {
			return resp
		}

		log.Printf("s3 request failure: %s, retry in %s", err, delay)
		time.Sleep(delay)

		delay = delay * 2
		if delay > 30*time.Second {
			delay = 30 * time.Second
		}
	}
}

// func (w *S3Writer) shipParts(rawParts []*bytes.Buffer) error {
// 	key := fmt.Sprintf("%d/%s.%s.txt", time.Now().Minute(), w.prefix, time.Now().Format(time.RFC3339))

// 	multi := retryRequest(func() (interface{}, error) {
// 		return w.bucket.InitMulti(key, "text/plain", s3.Private, s3.Options{})
// 	}).(*s3.Multi)

// 	completedParts := make(chan s3.Part, len(rawParts))
// 	for i, p := range rawParts {
// 		go func(n int, raw *bytes.Buffer) {
// 			completedParts <- retryRequest(func() (interface{}, error) {
// 				return multi.PutPart(n, bytes.NewReader(raw.Bytes()))
// 			}).(s3.Part)
// 		}(i+1, p)
// 	}

// 	var parts []s3.Part
// 	for part := range completedParts {
// 		parts = append(parts, part)
// 	}

// 	retryRequest(func() (interface{}, error) {
// 		return nil, multi.Complete(parts)
// 	})

// 	return nil
// }

// func (w *S3Writer) Run() error {
// 	var rawParts []*bytes.Buffer
// 	var rawPart *bytes.Buffer = bytes.NewBuffer(make([]byte, 0, PART_SIZE))

// 	for obj := range w.in {
// 		if rawPart.Len()+len(obj)+1 > PART_SIZE {
// 			rawParts = append(rawParts, rawPart)
// 			rawPart = bytes.NewBuffer(make([]byte, 0, PART_SIZE))
// 		}

// 		if len(rawParts) >= PART_COUNT {
// 			err := w.shipParts(rawParts)
// 			if err != nil {
// 				return fmt.Errorf("s3writer failed to ship parts: %s", err)
// 			}
// 			rawParts = []*bytes.Buffer{}
// 		}

// 		_, err := fmt.Fprintln(rawPart, string(obj))
// 		if err != nil {
// 			return fmt.Errorf("s3writer failed to write to buffer: %s", err)
// 		}
// 	}
// 	return fmt.Errorf("not implemented")
// }
