package main

import (
	"log"
	"sync"
	"sync/atomic"
	"time"
)

// periodically logs namespaced rate stats (n/sec)
type StatsLogger interface {
	// increments stat by specified value
	Add(stat string, n int64)

	// always call stop when you're done
	Stop()
}

type statsLogger struct {
	sync.RWMutex
	counts map[string]*int64
	done   chan struct{}

	namespace string
}

func NewStatsLogger(namespace string) StatsLogger {
	l := &statsLogger{
		counts:    make(map[string]*int64),
		done:      make(chan struct{}),
		namespace: namespace,
	}

	go l.run()

	return l
}

// looks complex but all it does is create stats if needed
func (s *statsLogger) Add(stat string, n int64) {
	var count *int64
	var ok bool

	s.RLock()
	count, ok = s.counts[stat]
	if ok {
		defer s.RUnlock()
	} else {
		s.RUnlock()
		s.Lock()
		defer s.Unlock()
		count = new(int64)
		s.counts[stat] = count
	}

	atomic.AddInt64(count, n)
}

func (s *statsLogger) Stop() {
	close(s.done)
}

func (s *statsLogger) fetchStats() map[string]int64 {
	stats := make(map[string]int64)

	s.RLock()
	defer s.RUnlock()

	for k, v := range s.counts {
		stats[k] = atomic.LoadInt64(v)
	}

	for k, v := range stats {
		atomic.AddInt64(s.counts[k], -v)
	}

	return stats
}

func (s *statsLogger) logStats() {
	stats := s.fetchStats()

	for k, v := range stats {
		log.Printf("%s.%s=%d", s.namespace, k, v)
	}
}

func (s *statsLogger) run() {

	ticker := time.NewTicker(1 * time.Second)
	defer ticker.Stop()

	for {
		select {
		case <-s.done:
		case <-ticker.C:
			s.logStats()
		}
	}
}
