package main

import (
	"bytes"
	"compress/gzip"
	"encoding/json"
	"fmt"
	"log"
	"strings"

	"code.justin.tv/ids/nfconvert/parse"
	"github.com/crowdmob/goamz/aws"
	"github.com/crowdmob/goamz/s3"
	"github.com/crowdmob/goamz/sqs"
)

// read in records and return a []byte of gzipped JSON records
// representing the same records
func gzjsonify(scanner *parse.RecordScanner) ([]byte, error) {
	var (
		buffer  bytes.Buffer
		newline = []byte{'\n'}
		err     error
	)
	gzBuffer, _ := gzip.NewWriterLevel(&buffer, gzip.BestSpeed)
	defer gzBuffer.Close()
	err = scanner.Map(func(r *parse.Record) error {
		b, err := json.Marshal(r.Abbreviate())
		if err != nil {
			return err
		}
		gzBuffer.Write(b)
		gzBuffer.Write(newline)
		return nil
	})
	if err != nil {
		return nil, err
	}
	return buffer.Bytes(), nil
}

func convert(inKey, outKey string, bucket *s3.Bucket) error {
	log.Printf("Reading %s\n", inKey)
	rc, err := bucket.GetReader(inKey)
	if err != nil {
		return err
	}
	defer rc.Close()

	log.Printf("Scanning %s\n", inKey)
	var scanner parse.RecordScanner
	if err = scanner.Init(rc, inKey); err != nil {
		return err
	}

	buf, err := gzjsonify(&scanner)
	if err != nil {
		return err
	}
	log.Printf("uploading %s (%d bytes)\n", outKey, len(buf))
	err = bucket.Put(
		outKey,
		buf,
		"binary/octet-stream",
		s3.BucketOwnerFull,
		s3.Options{},
	)
	if err != nil {
		log.Fatal(err)
	}
	log.Printf("Done!")

	return nil
}

func exists(key string, bucket *s3.Bucket) (bool, error) {
	resp, err := bucket.Head(key, map[string][]string{})
	if err != nil {
		return false, err
	}
	if resp == nil {
		return false, fmt.Errorf("nil response received")
	}
	if resp.StatusCode == 200 {
		return true, nil
	} else if resp.StatusCode == 404 {
		return false, nil
	} else {
		return false, fmt.Errorf("error response checking key, status code %d", resp.StatusCode)
	}
}

type handler struct {
	bucket *s3.Bucket
}

func (h *handler) init(auth aws.Auth, region aws.Region) {
	log.Printf("connecting to S3\n")
	s3client := s3.New(auth, region)
	h.bucket = s3client.Bucket(bucket)
}

func (h *handler) Handle(m *sqs.Message) error {
	inKey := m.Body
	outKey := outPrefix + strings.TrimPrefix(inKey, inPrefix) + ".json.gz"
	log.Printf("converting %s -> %s", inKey, outKey)

	convert(inKey, outKey, h.bucket)
	return nil
}

type benchmarkHandler struct {
	bucket *s3.Bucket
}

func (h *benchmarkHandler) init(auth aws.Auth, region aws.Region) {
	log.Printf("connecting to S3\n")
	s3client := s3.New(auth, region)
	h.bucket = s3client.Bucket(bucket)
}

func (h *benchmarkHandler) Handle(m *sqs.Message) error {
	inKey := m.Body
	outKey := outPrefix + strings.TrimPrefix(inKey, inPrefix) + ".json.gz"
	log.Printf("converting %s -> %s", inKey, outKey)

	convert(inKey, outKey, h.bucket)
	return nil
}
