package main

// Server listening on an SQS queue to convert s3 files
import (
	"flag"
	"log"
	"os"
	"os/signal"
	"runtime/pprof"
	"syscall"
	"time"

	_ "expvar"
	"net/http"
	_ "net/http/pprof"

	"github.com/crowdmob/goamz/aws"
	"github.com/crowdmob/goamz/s3"
	"github.com/twitchscience/aws_utils/listener"
)

const (
	inPrefix        = "data/raw/"
	outPrefix       = "data/json/"
	bucket          = "ids-netflows"
	sqsPollInterval = 10 * time.Second
	region          = "us-west-2"
)

func main() {
	var (
		secret       string
		access       string
		token        string
		sqsQueueName string
		concurrency  int
		perf         bool
	)
	flag.StringVar(&sqsQueueName, "sqs-queue", "ids-netflow-conversions", "sqs address to listen on")
	flag.IntVar(&concurrency, "n", 1, "number of concurrent workers to run")
	flag.BoolVar(&perf, "perf", false, "run once for perf")
	flag.StringVar(&secret, "secret-key", "", "aws_secret_access_key")
	flag.StringVar(&access, "access-key", "", "aws_access_key_id")
	flag.StringVar(&token, "token", "", "aws_secret_token")
	flag.Parse()

	auth, err := aws.GetAuth(access, secret, token, time.Now().Add(time.Hour*1))
	if err != nil {
		log.Fatal(err)
	}

	log.Printf("launching with access key id %s\n", auth.AccessKey)

	if perf {
		go func() {
			log.Println(http.ListenAndServe("localhost:6060", nil))
		}()
		log.Println("running benchmark")
		err := benchmark(auth)
		if err != nil {
			log.Fatal(err)
		}
		os.Exit(0)
	}

	log.Println("making workers")
	i := 0
	for i < concurrency {
		log.Println("Making worker", i)
		go work(sqsQueueName, auth)
		i += 1
	}

	sigCh := make(chan os.Signal, 1)
	signal.Notify(sigCh, syscall.SIGINT, syscall.SIGHUP)
	log.Println("Listening for signals...")
	for {
		switch <-sigCh {
		case syscall.SIGINT:
			os.Exit(0)
		case syscall.SIGHUP:
			os.Exit(0)
		}
	}
}

func work(queue string, auth aws.Auth) {
	log.Println("worker spawning")
	sqsAddr := listener.SQSAddr{
		aws.Regions["us-west-2"],
		queue,
		auth,
	}
	handler := handler{}
	handler.init(auth, aws.Regions[region])
	listener := listener.BuildSQSListener(&sqsAddr, &handler, sqsPollInterval)
	defer listener.Close()
	listener.Listen()
}

func benchmark(auth aws.Auth) error {
	f, err := os.Create("/tmp/nfconv.prof")
	if err == nil {
		pprof.StartCPUProfile(f)
		defer pprof.StopCPUProfile()
	}
	inKey := "data/raw/sfo01-br02/2015/02/05/nfcapd.201502050455"
	outKey := "dev/json/sfo01-br02/2015/02/05/nfcapd.201502050455.json.gz"
	log.Printf("connecting to S3\n")
	s3client := s3.New(auth, aws.Regions[region])
	bucket := s3client.Bucket(bucket)
	return convert(inKey, outKey, bucket)
}
