package main

import (
	"bufio"
	"encoding/json"
	"flag"
	"fmt"
	"log"
	"net"
	"os"
	"path/filepath"
	"time"

	"code.justin.tv/ids/nfconvert/parse"
	"code.justin.tv/ids/nfconvert/query"
)

var (
	dir  string
	port string = "23456"
	ip   string = "127.0.0.1"
)

func main() {
	flag.StringVar(&port, "port", port, "port to listen on")
	flag.StringVar(&ip, "ip", ip, "ip to bind to")
	flag.StringVar(&dir, "dir", "/tmp/results", "directory to write results into")
	flag.Parse()
	results := make(chan *query.QueryResult, 50)
	errors := make(chan error)
	go serve(results, errors)
	go write(results, errors)
	for {
		e := <-errors
		if e != nil {
			log.Fatal(e)
		}
	}
}

func write(results chan *query.QueryResult, errors chan error) {
	writer := NewResultWriter()
	defer writer.close()
	for {
		qr := <-results
		err := writer.write(qr)
		if err != nil {
			errors <- fmt.Errorf("write: %v", err)
		}
	}
}

func serve(results chan *query.QueryResult, errors chan error) {
	ln, err := net.Listen("tcp", ip+":"+port)
	if err != nil {
		log.Println(fmt.Errorf("tcp listen: %v", err))
		errors <- fmt.Errorf("tcp listen: %v", err)
	}
	log.Println("listening on " + ip + ":" + port)
	for {
		conn, err := ln.Accept()
		if err != nil {
			errors <- fmt.Errorf("tcp accept: %v", err)
		}
		go handle(conn, results, errors)
	}
}

func handle(c net.Conn, results chan *query.QueryResult, errors chan error) {
	scanner := bufio.NewScanner(c)
	for scanner.Scan() {
		qr := &query.QueryResult{}
		err := json.Unmarshal(scanner.Bytes(), qr)
		if err != nil {
			errors <- fmt.Errorf("Unparseable json: %v", err)
		}
		if err := scanner.Err(); err != nil {
			errors <- fmt.Errorf("Scanning error: %v", err)
		}
		results <- qr
	}
	if err := scanner.Err(); err != nil {
		errors <- err
	}
}

type ResultWriter struct {
	files map[string]*os.File
}

func NewResultWriter() *ResultWriter {
	return &ResultWriter{
		make(map[string]*os.File),
	}
}

func (w *ResultWriter) filename(name string) string {
	return filepath.Join(dir, fmt.Sprintf("query-result-%s", name))
}

func (w *ResultWriter) getfile(name string) (*os.File, error) {
	fname := w.filename(name)
	f, ok := w.files[fname]
	if !ok {
		var err error
		w.files[fname], err = os.Create(fname)
		return w.files[fname], err
	}
	return f, nil
}

func (w *ResultWriter) write(result *query.QueryResult) error {
	file, err := w.getfile(result.QueryName)
	if err != nil {
		return err
	}
	file.Write([]byte(String(result.FlowRecord) + "\n"))
	return nil
}

func (w *ResultWriter) close() {
	for _, f := range w.files {
		f.Close()
	}
}

func String(r *parse.Record) string {
	start := time.Unix(0, int64(r.Base.First)*1e9+int64(r.Base.MsecFirst)*1e6)
	end := time.Unix(0, int64(r.Base.Last)*1e9+int64(r.Base.MsecLast)*1e6)

	var (
		src = fmt.Sprintf("%s:%d", r.Base.SrcIP.String(), r.Base.SrcPort)
		dst = fmt.Sprintf("%s:%d", r.Base.DstIP.String(), r.Base.DstPort)
	)

	return fmt.Sprintf("%s  %dns  %s  ->  %s  %d  %d",
		start.Format(time.RFC3339),
		end.Sub(start),
		src,
		dst,
		r.Base.BytesIn,
		r.Base.Prot,
	)
}
