package parse

import (
	"encoding/binary"
	"io"
	"log"
	"net"
)

const (
	// IDs of extensions
	CommonExtID              = 0
	IPBlockExtID             = 1
	PacketCounterExtID       = 2
	InByteCounterExtID       = 3
	InterfaceRecord16ExtID   = 4
	InterfaceRecord32ExtID   = 5
	ASRecord16ExtID          = 6
	ASRecord32ExtID          = 7
	Extension8ExtID          = 8
	IPv4NextHopExtID         = 9
	IPv6NextHopExtID         = 10
	BGPNextHopIPv4ExtID      = 11
	BGPNextHopIPv6ExtID      = 12
	VLANRecordExtID          = 13
	PacketsOutCounter32ExtID = 14
	PacketsOutCounter64ExtID = 15
	BytesOutCounter32ExtID   = 16
	BytesOutCounter64ExtID   = 17
	AggrFlowsCounter32ExtID  = 18
	AggrFlowsCounter64ExtID  = 19
	MACOutboundExtID         = 20
	MACInboundExtID          = 21
)

func ReadExtension(r io.Reader, extID int, rec *Record) error {
	switch extID {
	case CommonExtID: // noop
	case InterfaceRecord16ExtID:
		ir, err := ReadInterfaceRecord(r, false)
		if err != nil {
			return err
		}
		rec.InterfaceRecord = ir
	case InterfaceRecord32ExtID:
		ir, err := ReadInterfaceRecord(r, true)
		if err != nil {
			return err
		}
		rec.InterfaceRecord = ir
	case ASRecord16ExtID:
		asr, err := ReadASRecord(r, false)
		if err != nil {
			return err
		}
		rec.ASRecord = asr
	case ASRecord32ExtID:
		asr, err := ReadASRecord(r, true)
		if err != nil {
			return err
		}
		rec.ASRecord = asr
	case Extension8ExtID:
		e8, err := ReadExtension8(r)
		if err != nil {
			return err
		}
		rec.DstTypeOfService = &e8.DstTos
		rec.Direction = &e8.Dir
		rec.SrcMask = &e8.SrcMask
		rec.DstMask = &e8.DstMask
	case IPv4NextHopExtID:
		ip, err := readIP(r, false)
		if err != nil {
			return err
		}
		rec.NextHop = ip
	case IPv6NextHopExtID:
		ip, err := readIP(r, true)
		if err != nil {
			return err
		}
		rec.NextHop = ip
	case BGPNextHopIPv4ExtID:
		ip, err := readIP(r, false)
		if err != nil {
			return err
		}
		rec.NextHop = ip
	case BGPNextHopIPv6ExtID:
		ip, err := readIP(r, true)
		if err != nil {
			return err
		}
		rec.NextHop = ip
	case VLANRecordExtID:
		vl, err := ReadVLAN(r)
		if err != nil {
			return err
		}
		rec.VLANRecord = vl
	case PacketsOutCounter32ExtID:
		n, err := ReadCounter(r, false)
		if err != nil {
			return err
		}
		rec.OutPackets = n
	case PacketsOutCounter64ExtID:
		n, err := ReadCounter(r, true)
		if err != nil {
			return err
		}
		rec.OutPackets = n
	case BytesOutCounter32ExtID:
		n, err := ReadCounter(r, false)
		if err != nil {
			return err
		}
		rec.OutBytes = n
	case BytesOutCounter64ExtID:
		n, err := ReadCounter(r, true)
		if err != nil {
			return err
		}
		rec.OutBytes = n
	case AggrFlowsCounter32ExtID:
		n, err := ReadCounter(r, false)
		if err != nil {
			return err
		}
		rec.AggrFlows = n
	case AggrFlowsCounter64ExtID:
		n, err := ReadCounter(r, true)
		if err != nil {
			return err
		}
		rec.AggrFlows = n
	case MACOutboundExtID:
		m1, err := ReadMACRecord(r)
		if err != nil {
			return err
		}
		rec.InSrcMAC = m1
		m2, err := ReadMACRecord(r)
		if err != nil {
			return err
		}
		rec.OutDstMAC = m2
	case MACInboundExtID:
		m1, err := ReadMACRecord(r)
		if err != nil {
			return err
		}
		rec.InDstMAC = m1
		m2, err := ReadMACRecord(r)
		if err != nil {
			return err
		}
		rec.OutSrcMAC = m2
	default:
		log.Printf("Unknown extension ID %d, skipping", extID)
	}
	return nil
}

/* Optional Extensions */

// Extension 4 / 5: Interface records
type InterfaceRecord struct {
	InputInterface  uint32
	OutputInterface uint32
}

// Same as above, but uses less space.
type ir16bit struct {
	InputInterface  uint16
	OutputInterface uint16
}

func ReadInterfaceRecord(r io.Reader, long bool) (*InterfaceRecord, error) {
	ifr := &InterfaceRecord{}
	if long {
		if err := binary.Read(r, binary.LittleEndian, ifr); err != nil {
			return nil, err
		}
	} else {
		temp := &ir16bit{}
		if err := binary.Read(r, binary.LittleEndian, temp); err != nil {
			return nil, err
		}
		ifr.InputInterface = uint32(temp.InputInterface)
		ifr.OutputInterface = uint32(temp.OutputInterface)
	}
	return ifr, nil
}

/* Extension 6: AS record, 16 bit */
type ASRecord struct {
	SrcAS uint32
	DstAS uint32
}
type asr16bit struct {
	SrcAS uint16
	DstAS uint16
}

func ReadASRecord(r io.Reader, long bool) (*ASRecord, error) {
	asr := &ASRecord{}
	if long {
		if err := binary.Read(r, binary.LittleEndian, asr); err != nil {
			return nil, err
		}
	} else {
		temp := &asr16bit{}
		if err := binary.Read(r, binary.LittleEndian, temp); err != nil {
			return nil, err
		}
		asr.SrcAS = uint32(temp.SrcAS)
		asr.DstAS = uint32(temp.DstAS)
	}
	return asr, nil
}

type Extension8 struct {
	DstTos  uint8
	Dir     uint8
	SrcMask uint8
	DstMask uint8
}

// Wombo combo extension. Returns these values:
//   dstToS:    Type of Service byte setting when exiting outgoing
//              interface
//   direction: 0 is ingress flow, 1 is egress flow
//   srcMask:   The number of contiguous bits in the source address
//              subnet mask i.e.: the submask in slash notation
//   dstMask:   The number of contiguous bits in the destination
//              address subnet mask i.e.: the submask in slash notation
func ReadExtension8(r io.Reader) (*Extension8, error) {
	ext8 := &Extension8{}
	if err := binary.Read(r, binary.LittleEndian, ext8); err != nil {
		return nil, err
	}
	return ext8, nil
}

func readIP(r io.Reader, ipv6 bool) (net.IP, error) {
	if ipv6 {
		return readIPv6Addr(r)
	} else {
		return readIPv4Addr(r)
	}
}

func ReadIPNextHop(r io.Reader, ipv6 bool) (net.IP, error) {
	return readIP(r, ipv6)
}

func ReadBGPNextHop(r io.Reader, ipv6 bool) (net.IP, error) {
	return readIP(r, ipv6)
}

type VLANRecord struct {
	SrcVLAN uint16
	DstVLAN uint16
}

// Extension 13: VLAN record
func ReadVLAN(r io.Reader) (*VLANRecord, error) {
	vlr := &VLANRecord{}
	if err := binary.Read(r, binary.LittleEndian, vlr); err != nil {
		return nil, err
	}
	return vlr, nil
}

type MACRecord [6]uint8

func ReadMACRecord(r io.Reader) (*MACRecord, error) {
	bytes := [6]uint8{}
	if err := binary.Read(r, binary.LittleEndian, &bytes); err != nil {
		return nil, err
	}
	mac := &MACRecord{}
	// Reverse the array
	for i := range bytes {
		mac[i] = bytes[5-i]
	}
	return mac, nil

}
