package parse

import (
	"encoding/binary"
	"fmt"
	"io"
)

const (
	ExtensionMapInitId = 0xFFFF
	MaxExtensions      = 65536
)

// ExtensionMaps
// =============
//
// ExtensionMaps contain information on the position of extensions
// within Records. They have an 8-byte header, followed by 0 or more
// 4-byte extension descriptions.
//
// The header:
//   The header has 4 fields, each 2 bytes:
//     1. record type (should always be set to 2)
//     2. size
//     3. MapID, referenced by records
//     4. MaxExtensionSize
//
// The descriptions:
//   Each extension is identified with an ID.

type ExtensionMap struct {
	*RecordHeader
	extMapHeader
	extensions []uint16
}

func ReadExtensionMap(r io.Reader, rh *RecordHeader) (*ExtensionMap, error) {
	var err error

	// First, parse header separately, since it is of fixed size but
	// the final ExtensionMap is variable size.
	h := extMapHeader{}
	if err = binary.Read(r, binary.LittleEndian, &h); err != nil {
		return nil, err
	}

	// 8 byte header, 2 bytes per extension:
	nExtensions := (rh.Size - 8) / 2
	m := &ExtensionMap{rh, h, make([]uint16, nExtensions)}

	// We expect to get an io.EOF error if everything goes smoothly
	i := 0
	for err == nil {
		var eid uint16
		err = binary.Read(r, binary.LittleEndian, &eid)
		if err == nil {
			m.extensions[i] = eid
			i += 1
		}
	}
	if err != nil && err != io.EOF {
		return nil, err
	}
	if i != int(nExtensions) {
		return nil, fmt.Errorf("Expected %d extensions but only found %d", nExtensions, i)
	}
	return m, nil

}

type extMapHeader struct {
	MapID         uint16
	ExtensionSize uint16
}

// Helper for parsing binary
type extensionDescriptor struct {
	ID     uint16
	Offset uint16
}

func (m *ExtensionMap) Set(id uint16, offset uint16) {
	m.extensions[id] = offset
}

func (m *ExtensionMap) GetOffsetOf(id uint16) uint16 {
	return m.extensions[id]
}
