package parse

// file.go describes how to read file-level attributes: the File header and the stat record

// Generic file handler
type NFFile struct {
	fileHeader  *FileHeader
	blockHeader *DataBlockHeader
	statRecord  *StatRecord

	compressed int
	fd         int
}

/*
 * nfdump binary file layout
 * =========================
 * Each data file starts with a file header, which identifies the file
 * as an nfdump data file.  The magic 16bit integer at the beginning
 * of each file must read 0xA50C. This also guarantees that endian
 * dependant files are read correct.
 *
 * Principal layout, recognized as LAYOUT_VERSION_1:
 *
 *   +-----------+-------------+-------------+-------------+-----+-------------+
 *   |Fileheader | stat record | datablock 1 | datablock 2 | ... | datablock n |
 *   +-----------+-------------+-------------+-------------+-----+-------------+
 */
const (
	// Magic to recognize nfdump file type and endianness
	MAGIC uint16 = 0xA50C
	// Version of binary file layout
	LAYOUT_VERSION_1 = 1

	NUM_FLAGS = 3
	// Flow records are compressed with LZO1X-1
	FLAG_COMPRESSED = 1 // LZO1X-1
	// Flow data are anonymized
	FLAG_ANONYMIZED = 2
	// Has a file catalog record after stat record
	FLAG_CATALOG = 4
)

type FileHeader struct {
	Magic      uint16
	Version    uint16
	Flags      uint32
	NumBlocks  uint32
	IdentBytes [128]byte
}

func (h *FileHeader) IdentString() string {
	n := 0
	for _, b := range h.IdentBytes {
		if b != 0 {
			n += 1
		}
	}
	return string(h.IdentBytes[:n])
}

func (h *FileHeader) validate() bool {
	return h.Magic == MAGIC
}

func (h *FileHeader) Compressed() bool {
	return (h.Flags & FLAG_COMPRESSED) != 0
}

func (h *FileHeader) Anonymized() bool {
	return (h.Flags & FLAG_ANONYMIZED) != 0
}

// StatRecord provides aggregate information about all records in the file.
type StatRecord struct {
	// overall
	NumFlows   uint64
	NumBytes   uint64
	NumPackets uint64

	// net stats
	NumFlowsTCP   uint64
	NumFlowsUDP   uint64
	NumFlowsICMP  uint64
	NumFlowsOther uint64

	// byte stats
	NumBytesTCP   uint64
	NumBytesUDP   uint64
	NumBytesICMP  uint64
	NumBytesOther uint64

	// packet stat
	NumPacketsTCP   uint64
	NumPacketsUDP   uint64
	NumPacketsICMP  uint64
	NumPacketsOther uint64

	// time window
	FirstSeen uint32
	LastSeen  uint32
	MsecFirst uint16
	MsecLast  uint16

	// other
	SequenceFailure uint32
}

type StatHeader struct {
	// Stat record type
	Type uint16
	// Size of the stat record in bytes without this header
	Size uint16
}
