package parse

import (
	"encoding/binary"
	"io"
	"net"
)

type ipv4 [4]uint8

func (ip *ipv4) toNetIP() net.IP {
	out := make(net.IP, 4)
	for i := 0; i < 4; i++ {
		out[i] = ip[3-i]
	}
	return out
}

type ipv6 [16]uint8

func (ip *ipv6) toNetIP() net.IP {
	out := make(net.IP, 16)
	for i := 0; i < 16; i++ {
		out[i] = ip[15-i]
	}
	return out
}

type ipv4Block struct {
	Src ipv4
	Dst ipv4
}

type ipv6Block struct {
	Src ipv6
	Dst ipv6
}

// Read IPv4 addresses from nfdump record bytes
func readIPv4AddrBlock(r io.Reader) (src net.IP, dst net.IP, err error) {
	ips := &ipv4Block{}
	if err := binary.Read(r, binary.LittleEndian, ips); err != nil {
		return nil, nil, err
	}
	src = ips.Src.toNetIP()
	dst = ips.Dst.toNetIP()
	return src, dst, nil
}

// Read IPv6 addresses from nfdump record bytes
func readIPv6AddrBlock(r io.Reader) (src net.IP, dst net.IP, err error) {
	ips := &ipv6Block{}
	if err := binary.Read(r, binary.LittleEndian, ips); err != nil {
		return nil, nil, err
	}
	src = ips.Src.toNetIP()
	dst = ips.Dst.toNetIP()
	return src, dst, nil
}

func readIPv4Addr(r io.Reader) (net.IP, error) {
	ip := &ipv4{}
	if err := binary.Read(r, binary.LittleEndian, ip); err != nil {
		return nil, err
	}
	return ip.toNetIP(), nil
}

func readIPv6Addr(r io.Reader) (net.IP, error) {
	ip := &ipv6{}
	if err := binary.Read(r, binary.LittleEndian, ip); err != nil {
		return nil, err
	}
	return ip.toNetIP(), nil
}
