package parse

import (
	"encoding/binary"
	"encoding/json"
	"fmt"
	"io"
	"net"
)

// This file contains structs describing individual records in an nfdump file.

// All records are prepended with a header
type RecordHeader struct {
	Type uint16
	Size uint16
}

// Record Types
// TODO: Implement more than just RecordType 10 and 2
const (
	CommonRecordV0Type = 1
	ExtensionMapType   = 2
	PortHistogramType  = 3
	BppHistogramType   = 4

	ExporterRecordType = 5
	SamplerRecordType  = 6

	ExporterInfoRecordType = 7
	ExporterStatRecordType = 8
	SamplerInfoRecordType  = 9

	CommonRecordType = 10
)

func ReadRecordHeader(r io.Reader) (*RecordHeader, error) {
	h := &RecordHeader{}

	if err := binary.Read(r, binary.LittleEndian, h); err != nil {
		return nil, err
	}

	return h, nil
}

type CommonRecord struct {
	commonRecordHeader

	exmap *ExtensionMap

	SrcIP net.IP
	DstIP net.IP

	PacketsIn uint64
	BytesIn   uint64
}

func NewCommonRecord(h *commonRecordHeader, m *ExtensionMap) *CommonRecord {
	return &CommonRecord{*h, m, nil, nil, 0, 0}
}

type commonRecordHeader struct {
	Flags  uint16
	ExtMap uint16

	MsecFirst uint16
	MsecLast  uint16
	First     uint32
	Last      uint32

	FwdStatus uint8
	TcpFlags  uint8
	Prot      uint8
	Tos       uint8
	SrcPort   uint16
	DstPort   uint16

	ExporterSysid uint16
	Reserved      uint16
}

// Common Record Flags
const (
	FlagIpv6Addr = 0x01
	FlagPkg64    = 0x02
	FlagBytes64  = 0x04
	FlagIpv6Nh   = 0x08
	FlagIpv6Nhb  = 0x0F
	FlagIpv6Exp  = 0x10
	FlagEvent    = 0x20
	FlagSampled  = 0x40
)

// ReadCommonRecord producesa a CommonRecord from the bytes in r. It
// uses exmap to look up the ExtensionMap which contains information
// on how to parse the variable-length trailer.
func ReadCommonRecord(r io.Reader, exmap map[uint16]*ExtensionMap) (*CommonRecord, error) {
	h := &commonRecordHeader{}

	if err := binary.Read(r, binary.LittleEndian, h); err != nil {
		return nil, err
	}

	em := exmap[h.ExtMap]
	if em == nil {
		return nil, fmt.Errorf("Corrupt data file! No such extension map id: %v. Skip record %v", h.ExtMap, h)
	}

	rec := NewCommonRecord(h, em)
	if err := rec.readIPAddrs(r); err != nil {
		return nil, err
	}
	if err := rec.readCounters(r); err != nil {
		return nil, err
	}
	return rec, nil
}

func (cr *CommonRecord) readIPAddrs(r io.Reader) error {
	// IP addresses can be appended as either ipv4 or ipv6 after the
	//CommonRecord header. In either case, source IP comes before
	//destination IP. Flags set on the record header inform us whether
	//it's ipv4 (so we should read 4 bytes for each IP) or ipv6 (so we
	//should read 16 bytes for each).

	ipv6 := cr.Flags&FlagIpv6Addr != 0

	var err error
	if ipv6 {
		cr.SrcIP, cr.DstIP, err = readIPv6AddrBlock(r)
		if err != nil {
			return err
		}
	} else {
		cr.SrcIP, cr.DstIP, err = readIPv4AddrBlock(r)
		if err != nil {
			return err
		}
	}
	return nil
}

func (cr *CommonRecord) readCounters(r io.Reader) error {
	var err error
	// Packet count can be stored as either a 32 or 64 bit number,
	// signaled by a flag.
	bigPacketCount := cr.Flags&FlagPkg64 != 0

	cr.PacketsIn, err = ReadCounter(r, bigPacketCount)
	if err != nil {
		return err
	}

	// Byte count is similar
	bigByteCount := cr.Flags&FlagBytes64 != 0
	cr.BytesIn, err = ReadCounter(r, bigByteCount)
	if err != nil {
		return err
	}

	return nil
}

// Called a 'master record in nfdump, this is a struct which contains
// fields for everything that could be recorded.
type Record struct {
	Base            *CommonRecord
	InputInterface  *uint64
	OutputInterface *uint64
	InterfaceRecord *InterfaceRecord
	ASRecord        *ASRecord

	DstTypeOfService *uint8
	Direction        *uint8
	SrcMask          *uint8
	DstMask          *uint8

	NextHop    net.IP
	BGPNextHop net.IP
	VLANRecord *VLANRecord

	OutPackets uint64
	OutBytes   uint64
	AggrFlows  uint64

	InSrcMAC  *MACRecord
	OutDstMAC *MACRecord
	InDstMAC  *MACRecord
	OutSrcMAC *MACRecord
}

type BriefRecord struct {
	SrcIP     net.IP
	DstIP     net.IP
	SrcPort   uint16
	DstPort   uint16
	Packets   uint64
	Bytes     uint64
	Protocol  uint8
	Startns   uint64
	Endns     uint64
	InSrcMAC  *MACRecord
	OutDstMAC *MACRecord
}

func (r *Record) Abbreviate() *BriefRecord {
	return &BriefRecord{
		SrcIP:     r.Base.SrcIP,
		DstIP:     r.Base.DstIP,
		SrcPort:   r.Base.SrcPort,
		DstPort:   r.Base.DstPort,
		Packets:   r.Base.PacketsIn,
		Bytes:     r.Base.BytesIn,
		Protocol:  r.Base.Prot,
		InSrcMAC:  r.InSrcMAC,
		OutDstMAC: r.OutDstMAC,
		Startns:   uint64(r.Base.First)*1e9 + uint64(r.Base.MsecFirst)*1e6,
		Endns:     uint64(r.Base.Last)*1e9 + uint64(r.Base.MsecLast)*1e6,
	}
}

func (r *Record) GetSrcAS() *uint32 {
	if r.ASRecord == nil {
		return nil
	}
	return &r.ASRecord.SrcAS
}

func (r *Record) GetDstAS() *uint32 {
	if r.ASRecord == nil {
		return nil
	}
	return &r.ASRecord.DstAS
}

func (r *Record) GetSrcVLAN() *uint16 {
	if r.VLANRecord == nil {
		return nil
	}
	return &r.VLANRecord.SrcVLAN
}

func (r *Record) GetDstVLAN() *uint16 {
	if r.VLANRecord == nil {
		return nil
	}
	return &r.VLANRecord.DstVLAN
}

func NewRecord(base *CommonRecord) *Record {
	return &Record{Base: base}
}

func (rec *Record) ProcessExtensions(r io.Reader) error {
	for _, i := range rec.Base.exmap.extensions {
		err := ReadExtension(r, int(i), rec)
		if err != nil {
			return err
		}
	}
	return nil
}

func (rec *Record) ToJSON() string {
	b, _ := json.Marshal(rec)
	return fmt.Sprintf("%s", b)
}
