package query

import (
	"log"
	"net"

	"code.justin.tv/ids/nfconvert/parse"
	"github.com/crowdmob/goamz/sqs"
)

// Handlers are used by workers to process data

type Filter interface {
	Matches(*parse.Record) bool
	Marshal() FilterRPC
}

type FilterRPC struct {
	Name string
	Args map[string]string
}

func (rpc *FilterRPC) Unmarshal() Filter {
	switch rpc.Name {
	case "IPFilter":
		return &IPFilter{IP: net.ParseIP(rpc.Args["ip"])}
	case "LargeOutboundFlows":
		return &LargeOutboundFlows{}
	default:
		log.Fatal("Unknown handler was received: ", rpc.Name)
	}
	return nil
}

type IPFilter struct {
	IP      net.IP
	resultQ *sqs.Queue
	n       int
}

func (f *IPFilter) Marshal() FilterRPC {
	args := make(map[string]string)
	args["ip"] = f.IP.String()
	return FilterRPC{"IPFilter", args}
}

func (f *IPFilter) Matches(r *parse.Record) bool {
	return f.IP.Equal(r.Base.SrcIP) || f.IP.Equal(r.Base.DstIP)
}

var twitchSubnets []net.IPNet = []net.IPNet{
	net.IPNet{net.ParseIP("10.0.0.0"), net.CIDRMask(8, 32)},
	net.IPNet{net.ParseIP("192.16.64.0"), net.CIDRMask(21, 32)},
	net.IPNet{net.ParseIP("199.9.248.0"), net.CIDRMask(21, 32)},
	net.IPNet{net.ParseIP("185.42.204.0"), net.CIDRMask(22, 32)},
}

type LargeOutboundFlows struct {
	MinSize uint64
}

func (f *LargeOutboundFlows) Marshal() FilterRPC {
	args := make(map[string]string)
	return FilterRPC{"LargeOutboundFlows", args}
}

func (f *LargeOutboundFlows) Matches(r *parse.Record) bool {
	var (
		srcIsInternal bool = false
	)

	if r.Base.SrcPort == 80 || r.Base.SrcPort == 443 || r.Base.SrcPort == 1935 {
		return false
	}

	for _, net := range twitchSubnets {
		if net.Contains(r.Base.DstIP) {
			return false
		}
		if !srcIsInternal && net.Contains(r.Base.SrcIP) {
			srcIsInternal = true
		}
	}

	return srcIsInternal && r.Base.BytesIn > f.MinSize
}
