package query

import (
	"encoding/json"
	"log"
	"time"

	"code.justin.tv/ids/nfconvert/parse"
	"github.com/crowdmob/goamz/aws"
	"github.com/crowdmob/goamz/s3"
	"github.com/crowdmob/goamz/sqs"
	"github.com/twitchscience/aws_utils/listener"
)

func Work(auth aws.Auth, collectorAddr string) error {
	tp, err := newTaskProcessor(auth, collectorAddr)
	if err != nil {
		return err
	}
	listener := listener.BuildSQSListener(
		GetTaskQueueAddr(auth),
		tp,
		2*time.Second,
	)
	defer listener.Close()
	listener.Listen()
	return nil
}

type taskProcessor struct {
	// handlers for external calls
	taskQ         *sqs.Queue
	resultQ       *sqs.Queue
	bucket        *s3.Bucket
	collectorAddr string
	// stateful data per-query
	req     *QueryRequest
	f       Filter
	queryID uint32
}

func newTaskProcessor(auth aws.Auth, collector string) (*taskProcessor, error) {
	var err error
	tp := taskProcessor{}
	tp.taskQ, err = GetTaskQueue(auth)
	if err != nil {
		return nil, err
	}
	tp.resultQ, err = GetResultQueue(auth)
	if err != nil {
		return nil, err
	}

	tp.bucket = GetDataBucket(auth)
	tp.collectorAddr = collector
	return &tp, nil
}

// to implement listener.SQSHandler
func (tp *taskProcessor) Handle(m *sqs.Message) error {
	// Unmarshal request
	req := &QueryRequest{}
	err := json.Unmarshal([]byte(m.Body), req)
	if err != nil {
		return err
	}
	log.Printf("handling %s", req)

	tp.setRequest(req)
	// Set query ID
	if err := tp.process(); err != nil {
		log.Println("ERROR processing:", err)
	}
	tp.taskQ.DeleteMessage(m)
	return nil
}

func (tp *taskProcessor) setRequest(req *QueryRequest) {
	filter := req.Filter.Unmarshal()
	tp.f = filter
	tp.req = req
}

func (tp *taskProcessor) newResult() *QueryResult {
	qr := &QueryResult{
		S3Key:     tp.req.S3Key,
		QueryName: tp.req.QueryName,
	}
	return qr
}

func (tp *taskProcessor) processOne(r *parse.Record) error {
	if tp.f.Matches(r) {
		qr := tp.newResult()
		qr.FlowRecord = r
		return qr.Send(tp.collectorAddr)
	}
	return nil
}

func (tp *taskProcessor) process() error {
	log.Printf("Processing started")
	var (
		err     error
		key     string
		scanner parse.RecordScanner
	)
	key = tp.req.S3Key

	rc, err := tp.bucket.GetReader(key)
	if err != nil {
		return err
	}
	defer rc.Close()

	if err = scanner.Init(rc, key); err != nil {
		return err
	}
	err = scanner.Map(tp.processOne)
	if err != nil {
		return err
	}
	log.Printf("%d: done with %s\n", tp.req.QueryName, key)
	return nil
}
