package com.twitch.spark.queries;

import java.io.Serializable;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.UnknownHostException;

public class CIDRMask implements Serializable {
    private Inet4Address i4addr;
    private byte maskCtr;

    private int addrInt;
    private int maskInt;

    public CIDRMask(Inet4Address i4addr, byte mask)
    {
        this.i4addr = i4addr;
        this.maskCtr = mask;

        this.addrInt = addrToInt(i4addr);
        this.maskInt = ~((1 << (32 - maskCtr)) - 1);
    }

    /** IPMask factory method.
     *
     * @param addrSlashMask IP/Mask String in format "nnn.nnn.nnn.nnn/mask". If
     *    the "/mask" is omitted, "/32" (just the single address) is assumed.
     * @return a new IPMask
     * @throws UnknownHostException if address part cannot be parsed by
     *    InetAddress
     */
    public static CIDRMask getIPMask(String addrSlashMask) {
        int pos = addrSlashMask.indexOf('/');
        String addr;
        byte maskCtr;
        if (pos==-1) {
            addr = addrSlashMask;
            maskCtr = 32;
        } else {
            addr = addrSlashMask.substring(0, pos);
            maskCtr = Byte.parseByte(addrSlashMask.substring(pos + 1));
        }
        try {
            return new CIDRMask((Inet4Address) InetAddress.getByName(addr), maskCtr);
        } catch (UnknownHostException e) {
            return null;
        }
    }

    public boolean matches(Inet4Address testAddr) {
        int testAddrInt = addrToInt(testAddr);
        return ((addrInt & maskInt) == (testAddrInt & maskInt));
    }

    public boolean matches(String addr) throws UnknownHostException {
        return matches((Inet4Address)InetAddress.getByName(addr));
    }
    /** Converts IPv4 address to integer representation.
     */
    private static int addrToInt(Inet4Address i4addr)
    {
        byte[] ba = i4addr.getAddress();
        return     (ba[0]       << 24)
                | ((ba[1]&0xFF) << 16)
                | ((ba[2]&0xFF) << 8)
                |  (ba[3]&0xFF);
    }

}