package com.twitch.spark.queries;
import java.io.IOException;
import java.io.Serializable;
import java.net.InetAddress;
import java.util.List;



import com.fasterxml.jackson.databind.ObjectMapper;
import org.joda.time.DateTime;

public final class Netflow implements Serializable {
    private static final long serialVersionUID = 420L;
    private static final ObjectMapper jsonMapper = new ObjectMapper();

    public final long bytes;
    public final long packets;
    public final String srcIP;
    public final int srcPort;
    public final String dstIP;
    public final int dstPort;

    public final byte protocol;

    public final List<Byte> inSrcMAC;
    public final List<Byte> outDstMAC;

    public final DateTime start;
    public final DateTime end;

    public Netflow(long bytes, long packets, DateTime start, DateTime end,
                   String srcIP, int srcPort, String dstIP, int dstPort,
                   byte protocol, List<Byte> inSrcMAC, List<Byte> outDstMAC) {
        this.bytes = bytes;
        this.packets = packets;
        this.srcIP = srcIP;
        this.srcPort = srcPort;
        this.dstIP = dstIP;
        this.dstPort = dstPort;
        this.protocol = protocol;
        this.inSrcMAC = inSrcMAC;
        this.outDstMAC = outDstMAC;
        this.start = start;
        this.end = end;
    }

    // class to represent raw JSON data for a flow
    public static class NetflowJSON {
        public long Bytes;
        public long Packets;
        public String SrcIP;
        public String DstIP;
        public int SrcPort;
        public int DstPort;
        public byte Protocol;
        public long Startns;
        public long Endns;
        public List<Byte> InSrcMAC;
        public List<Byte> OutDstMAC;
    }

    // constructor from a JSON string
    public Netflow(String jsonData) throws IOException {
        NetflowJSON raw = jsonMapper.readValue(jsonData, NetflowJSON.class);
        this.bytes = raw.Bytes;
        this.packets = raw.Packets;
        this.srcIP = raw.SrcIP;
        this.dstIP = raw.DstIP;
        this.srcPort = raw.SrcPort;
        this.dstPort = raw.DstPort;
        this.protocol = raw.Protocol;
        this.start = new DateTime(raw.Startns / 1000000);
        this.end = new DateTime(raw.Endns / 1000000);
        this.inSrcMAC = raw.InSrcMAC;
        this.outDstMAC = raw.OutDstMAC;
    }


    @Override
    public String toString() {

        return "Netflow{" +
                "bytes=" + bytes +
                ", packets=" + packets +
                ", srcIP=" + srcIP +
                ", srcPort=" + srcPort +
                ", dstIP=" + dstIP +
                ", dstPort=" + dstPort +
                ", protocol=" + protocol +
                ", inSrcMAC=" + inSrcMAC +
                ", outDstMAC=" + outDstMAC +
                ", start=" + start +
                ", end=" + end +
                '}';
    }
}
