package com.twitch.spark.queries;

import com.google.common.collect.ImmutableList;
import org.apache.spark.api.java.function.Function;

import java.io.Serializable;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.UnknownHostException;

public class NetflowFilters implements Serializable {
    private static final ImmutableList<CIDRMask> TwitchIPBlocks = ImmutableList.of(
            CIDRMask.getIPMask("10.0.0.0/8"),
            CIDRMask.getIPMask("192.16.64.0/21"),
            CIDRMask.getIPMask("199.9.248.0/21"),
            CIDRMask.getIPMask("185.42.204.0/22")
    );

    public static final class FilterToOutboundTraffic implements Function<Netflow, Boolean> {
        public Boolean call(Netflow f) {
            Boolean srcIsInternal = false;
            Boolean dstIsInternal = false;
            try {
                for (int i = 0; i < TwitchIPBlocks.size(); i++) {
                    CIDRMask m = TwitchIPBlocks.get(i);
                    if (!srcIsInternal && m.matches(f.srcIP)) {
                        srcIsInternal = true;
                    }
                    if (!dstIsInternal && m.matches(f.dstIP)) {
                        dstIsInternal = true;
                    }
                }
            } catch (UnknownHostException e) {
                System.out.println(e.getMessage());
                return false;
            } catch (ClassCastException e) {
                System.out.println(e.getMessage());
                System.out.println(f);
                return false;
            }
            if (srcIsInternal && (f.srcPort == 80 || f.srcPort == 443)) {
                return false;
            }
            return srcIsInternal && !dstIsInternal;
        }
    }

    public static final class FilterNulls implements Function<Netflow, Boolean> {
        public Boolean call(Netflow f) {
            return f != null;
        }
    }

    public static final class FilterLargeTransfers implements Function<Netflow, Boolean> {
        public Boolean call(Netflow f) {
            return f.bytes > 20000000;  // 20mb
        }
    }

    public static final class FilterIPs implements Function<Netflow, Boolean> {
        private InetAddress ipFilter;

        public FilterIPs(InetAddress ipFilter) {
            this.ipFilter = ipFilter;
        }

        public Boolean call(Netflow f) {
            return (f.srcIP.equals(ipFilter) || f.dstIP.equals(ipFilter));
        }
    }
}
