package com.twitch.spark.queries;

import org.apache.spark.api.java.JavaSparkContext;
import org.apache.spark.api.java.function.Function;

import java.io.IOException;
import java.io.Serializable;
import java.util.Comparator;
import java.util.List;

public class Utilities {
    public static void setAuth(JavaSparkContext jsc, String accessKey, String secretKey) {
        jsc.hadoopConfiguration().set("fs.s3n.awsAccessKeyId", accessKey);
        jsc.hadoopConfiguration().set("fs.s3.awsAccessKeyId", accessKey);
        jsc.hadoopConfiguration().set("fs.s3n.awsSecretAccessKey", secretKey);
        jsc.hadoopConfiguration().set("fs.s3.awsSecretAccessKey", secretKey);
    }

    public static String inputPath(String path) {
        return "s3n://ids-netflows/data/json/" + path;
    }


    public static String outputPath(String path) {
        return "s3n://ids-netflows/data/queryresults/" + path;
    }


    public static String humanReadableByteCount(long bytes) {
        if (bytes < 1000) return bytes + " B";
        int exp = (int) (Math.log(bytes) / Math.log(1000));
        char pre = "kMGTPE".charAt(exp - 1);
        return String.format("%.1f %sB", bytes / Math.pow(1000, exp), pre);
    }

    public static void printFlows(List<Netflow> flows) {
        String header = "start                      proto  srcip:port                 dstip:port              bytes";
        String format = "%-24s  %3d  %15s:%-6d  ->  %15s:%-6d    %-9s\n";
        System.out.println(header);
        for (Netflow flow : flows) {
            System.out.format(format,
                    flow.start, flow.protocol, flow.srcIP, flow.srcPort, flow.dstIP, flow.dstPort, humanReadableByteCount(flow.bytes)
            );
        }
    }

    public static final class SortByBytesDesc implements Comparator<Netflow>, Serializable {
        public int compare(Netflow f1, Netflow f2) {

            if (f1.bytes < f2.bytes) {
                return 1;
            } else if (f1.bytes > f2.bytes) {
                return -1;
            } else {
                return 0;
            }
        }
    }

    public static final class ParseJSON implements Function<String, Netflow> {
        public Netflow call(String s) {
            try {
                return new Netflow(s);
            } catch(IOException e) {
                System.out.println("failure to parse " + s);
                System.out.println(e.getMessage());
                return null;
            }
        }
    }
}
