#!/usr/bin/env bash
set -eo pipefail

function show_help {
cat << EOF
Usage: nfupload.sh [--dev] SOURCE_DIR

Uploads all files matching the glob "nfcapd*" in SOURCE_DIR to 
s3://ids-netflows/data/raw/

You must set two environment variables for credentials: AWS_ACCESS_KEY_ID
and AWS_SECRET_KEY

If --dev is passed, files are synced to s3://ids-netflows/dev/data/raw instead.
EOF
}

case "$#" in
    0)
        # no arguments: exit w/ usage
        show_help
        exit 0
        ;;
    1)
        # 1 argument: its the source directory
        local=$1
        remote="s3://ids-netflows/data/raw/"
        ;;
    2)
        # 2 arguments: make sure first is '--dev', else exit
        if [[ "$1" != --dev ]]; then
            show_help
            exit 0
        fi
        local=$2
        remote="s3://ids-netflows/dev/data/raw/"
        ;;
    *)
        # >2 arguments: exit w/ usage
        show_help
        exit 0
        ;;
esac

key_id=$AWS_ACCESS_KEY_ID
secret=$AWS_SECRET_KEY

if [[ -z "$key_id" ]]; then
    echo "failed - AWS_ACCESS_KEY_ID variable unset"
    exit 1
fi

if [[ -z "$secret" ]]; then
    echo "failed - AWS_SECRET_KEY variable unset"
    exit 1
fi

# use ionice if its available
cmd=""
if hash ionice 2>/dev/null; then
    cmd="$cmd ionice -c 3"
fi

if hash nice 2>/dev/null; then
    cmd="$cmd nice -n 7"
fi
here=$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd ) # directory of script

$cmd $here/s3sync $key_id $secret $local $remote
