package main

import (
	"flag"
	"fmt"
	"os"

	"code.justin.tv/ids/s3sync"

	"github.com/jcelliott/lumber"
)

func usage() {
	fmt.Fprintf(os.Stderr, `USAGE

s3sync [options] SYNCFROM SYNCTO

Syncs directory SYNCTO to match contents of SYNCFROM. Both SYNCTO and SYNCFROM 
can be either local paths or remote S3 paths. S3 paths should be specified 
with this syntax:

  s3://<bucket>/<optional path>

For example,

  s3sync /var/log/ s3://logs/raw/

would update s3://logs/raw/ to match the contents of /var/log/, including traversing 
subdirectories.

OPTIONS
`)
	flag.PrintDefaults()
	fmt.Fprintf(os.Stderr, `
PERMISSIONS
If you don't have permission to read a local file, it will be skipped. To see 
when this happens, turn on verbose logging.
`)
}
func main() {
	flag.Usage = usage
	var (
		// Log levels
		quiet   bool
		verbose bool
		debug   bool
		// Paths to sync
		from string
		to   string
		// AWS auth
		awsKeyID  string
		awsSecret string
		awsRegion string
		// sync method
		deleteMissing bool
		skipChecksum  bool
		err           error
	)
	flag.BoolVar(&quiet, "quiet", false, "suppress all log output")
	flag.BoolVar(&verbose, "verbose", false, "be chatty in logs")
	flag.BoolVar(&debug, "debug", false, "be overwhelmingly chatty in logs")
	flag.BoolVar(&deleteMissing, "delete-missing", false, "delete files in SYNCTO if they are not present in SYNCFROM")
	flag.BoolVar(&skipChecksum, "skip-checksum", false, "skip checking md5sums of files and just use file size instead")
	flag.StringVar(&awsKeyID, "aws-access-key-id", "", "access key ID to use when talking to AWS")
	flag.StringVar(&awsSecret, "aws-secret-key", "", "secret key to use when talking to AWS")
	flag.StringVar(&awsRegion, "aws-region", "us-west-2", "region to use when talking to AWS")
	flag.Parse()

	if quiet {
		s3sync.Log.Level(lumber.FATAL)
	} else if verbose {
		s3sync.Log.Level(lumber.INFO)
	} else if debug {
		s3sync.Log.Level(lumber.DEBUG)
	} else {
		s3sync.Log.Level(s3sync.DEFAULT_LOG_LEVEL)
	}
	from = flag.Arg(0)
	to = flag.Arg(1)

	if from == "" {
		s3sync.Log.Fatal("no SYNCFROM directory provided")
		usage()
		os.Exit(1)
	}
	if to == "" {
		s3sync.Log.Fatal("no SYNCTO directory provided")
		usage()
		os.Exit(1)
	}

	s3sync.DeleteMissing = deleteMissing
	s3sync.SkipChecksums = skipChecksum

	s3sync.SetAuth(awsKeyID, awsSecret)
	if err = s3sync.SetRegion(awsRegion); err != nil {
		s3sync.Log.Fatal(err.Error())
		os.Exit(1)
	}

	err = s3sync.Sync(from, to)

	if err != nil {
		s3sync.Log.Fatal(err.Error())
		os.Exit(1)
	}
}
