package s3sync

import (
	"errors"
	"fmt"
	"io"
	"os"
	"path/filepath"
)

type localDirectory struct {
	path string
}

func (d *localDirectory) Contents() (map[string]file, error) {
	Log.Debug("getting local contents of %s", d.path)
	contents := make(map[string]file)
	filepath.Walk(d.path, func(path string, info os.FileInfo, err error) error {
		if err != nil {
			Log.Info("skipping %s: %v", path, err)
			return nil
		}
		relpath, err := filepath.Rel(d.path, path)
		if err != nil {
			Log.Error("error computing relative path: %v", err)
			return err
		}
		if info.IsDir() {
			Log.Debug("skipping directory %s", path)
			// not a file. ignore.
			return nil
		}
		f := newLocalFile(relpath, info, d)
		Log.Debug("adding file at %s: %+v", path, f)
		contents[relpath] = f
		return nil
	})
	return contents, nil
}

func (d *localDirectory) Create(subpath string, from file) error {
	fullPath := filepath.Join(d.path, subpath)
	Log.Info("creating local file at %s", fullPath)
	Log.Debug("ensuring directory exists")

	if err := os.MkdirAll(filepath.Dir(fullPath), 0777); err != nil {
		Log.Debug("error creating directories")
		return err
	}

	to, err := os.Create(fullPath)
	if err != nil {
		return err
	}
	defer to.Close()

	if err = from.Open(); err != nil {
		return err
	}
	defer from.Close()

	Log.Debug("copying data from %+v to local %+v", from, to)
	if _, err = io.Copy(to, from); err != nil {
		return err
	}
	return nil
}

func (d *localDirectory) Delete(subpath string) error {
	fullPath := filepath.Join(d.path, subpath)
	Log.Info("deleting local file at %s", fullPath)
	return os.Remove(fullPath)
}

func (d *localDirectory) Update(subpath string, contents file) error {
	Log.Info("updating local file at %s", filepath.Join(d.path, subpath))
	if err := d.Delete(subpath); err != nil {
		return err
	}
	return d.Create(subpath, contents)
}

func (d *localDirectory) Clean() error {
	Log.Info("cleaning up empty directories in %s", d.path)
	directories := make(map[string]uint)
	filepath.Walk(d.path, func(path string, info os.FileInfo, err error) error {
		Log.Debug("dir cleanup: scanning %s", path)
		if err != nil {
			Log.Debug("skipping %s: %v", path, err)
		}
		if info.IsDir() {
			Log.Debug("dir cleanup: %s is a directory - marking 0", path)
			if _, touched := directories[path]; !touched {
				directories[path] = 0
			}

		} else {
			Log.Debug("dir cleanup: %s is not a directory - incrementing", path)
			directories[filepath.Dir(path)] += 1
		}
		return nil
	})
	for dir, count := range directories {
		Log.Debug("dir cleanup: %s - %d", dir, count)
		if count == 0 {
			Log.Debug("removing %s", dir)
			os.Remove(dir)
		}
	}
	return nil
}

// make sure path is a directory
func validate(path string) error {
	if !exists(path) {
		return fmt.Errorf("%s does not exist", path)
	}
	stat, err := os.Stat(path)
	if err != nil {
		return err
	}
	if stat.IsDir() {
		return nil
	} else {
		return errors.New("not a directory")
	}
}

// make sure path exists
func exists(path string) bool {
	_, err := os.Stat(path)
	return !os.IsNotExist(err)
}
