package s3sync

import (
	"crypto/md5"
	"fmt"
	"io"
	"os"
	"path/filepath"
)

type localFile struct {
	directory    *localDirectory
	relativePath string // path relative to directory

	_size uint
	fd    *os.File
}

func newLocalFile(path string, info os.FileInfo, dir *localDirectory) *localFile {
	f := localFile{
		directory:    dir,
		relativePath: path,
		_size:        uint(info.Size()),
	}
	return &f
}

func (f *localFile) fullPath() string {
	return filepath.Join(f.directory.path, f.relativePath)
}

func (f *localFile) Open() error {
	var err error
	f.fd, err = os.Open(f.fullPath())
	if err != nil {
		return err
	}
	return nil
}

func (f *localFile) Close() error {
	return f.fd.Close()
}

func (f *localFile) Read(into []byte) (n int, err error) {
	return f.fd.Read(into)
}

func (f *localFile) Write(from []byte) (n int, err error) {
	return f.fd.Write(from)
}

func (f *localFile) size() uint {
	return f._size
}

// computes the md5 checksum of the file
func (f *localFile) checksum() (string, error) {
	err := f.Open()
	if err != nil {
		return "", fmt.Errorf("f.Open: %v", err)
	}
	defer f.Close()
	h := md5.New()
	if _, err = io.Copy(h, f); err != nil {
		return "", fmt.Errorf("f.Read: %v", err)
	}
	return fmt.Sprintf("%x", h.Sum(nil)), nil
}
