package s3sync

import (
	"fmt"
	"io/ioutil"
	"net/url"
	"strings"

	"gopkg.in/amz.v1/s3"
)

type s3Directory struct {
	bucket    *s3.Bucket
	keyPrefix string

	client *s3.S3
}

func newS3Directory(uri string) (*s3Directory, error) {
	Log.Debug("building s3directory struct for %+v in %+v", uri, awsRegion)

	bucketName, keyPrefix, err := splitS3URI(uri)
	if err != nil {
		return nil, err
	}
	// s3 doesn't expect a leading slash
	keyPrefix = strings.TrimPrefix(keyPrefix, "/")
	client := s3.New(awsAuth, awsRegion)
	bucket := client.Bucket(bucketName)
	d := s3Directory{
		bucket:    bucket,
		keyPrefix: keyPrefix,
		client:    client,
	}
	return &d, nil
}

// splits a URI which looks like s3://bucket/key/morekey into bucket, key/morekey
func splitS3URI(uri string) (bucketName, keyPrefix string, err error) {
	parsed, err := url.Parse(uri)
	if err != nil {
		return "", "", err
	}
	if parsed.Scheme != "s3" {
		return "", "", fmt.Errorf("expected s3:// scheme but got %s://", parsed.Scheme)
	}
	return parsed.Host, parsed.Path, nil
}

func (d *s3Directory) getFile() file {
	return &s3File{}
}

func (d *s3Directory) Contents() (map[string]file, error) {
	Log.Debug("getting s3 contents of %+v", d)
	contents := make(map[string]file)
	result, err := d.bucket.List(d.keyPrefix, "", "", 1000000)
	if err != nil {
		return nil, fmt.Errorf("bucket.List err: %v", err)
	}
	for _, key := range result.Contents {
		key.Key = strings.TrimPrefix(key.Key, d.keyPrefix)
		contents[key.Key] = s3FileFromKey(&key, d)
	}
	Log.Debug("found %d items in s3 directory %+v", len(contents), d)
	return contents, nil
}

func (d *s3Directory) Create(subpath string, from file) error {
	relPath := d.keyPrefix + subpath
	fullPath := fmt.Sprintf("s3://%s/%s", d.bucket.Name, relPath)
	Log.Info("creating s3 file at %s", fullPath)

	to := newEmptyS3File(subpath, d)

	Log.Debug("opening from-file %+v", from)
	if err := from.Open(); err != nil {
		return err
	}
	defer from.Close()

	Log.Debug("copying data from %+v to s3 %+v", from, to)
	// Can't use io.Copy here since it buffers and writes in chunks.
	if from.size() < multipart_threshold {
		buf, err := ioutil.ReadAll(from)
		if err != nil {
			return err
		}
		_, err = to.writeSimple(buf)
		return err
	} else {
		return to.streamMultipartWrite(from)
	}
}

func (d *s3Directory) Update(subpath string, contents file) error {
	if err := d.Delete(subpath); err != nil {
		return err
	}
	return d.Create(subpath, contents)
}

func (d *s3Directory) Delete(subpath string) error {
	relPath := d.keyPrefix + subpath
	Log.Info("deleting s3 file at %s", relPath)
	return d.bucket.Del(relPath)
}

func (d *s3Directory) Clean() error {
	return nil
}
