package s3sync

import (
	"bytes"
	"fmt"
	"io"
	"strings"
	"sync"

	"gopkg.in/amz.v1/s3"
)

const (
	multipart_part_size = 5 * 1024 * 1024 // 5MB
	multipart_threshold = 2 * multipart_part_size
)

// Represents a file residing in S3
type s3File struct {
	directory    *s3Directory
	relativePath string

	_size     uint
	_checksum string

	key *s3.Key

	rc io.ReadCloser
}

func newEmptyS3File(path string, d *s3Directory) *s3File {
	f := &s3File{
		directory:    d,
		relativePath: path,
	}
	return f
}

func s3FileFromKey(key *s3.Key, d *s3Directory) *s3File {
	f := &s3File{
		directory:    d,
		relativePath: key.Key,
		_size:        uint(key.Size),
		_checksum:    strings.Trim(key.ETag, "\""),
		key:          key,
	}
	return f
}

func (f *s3File) fullPath() string {
	return fmt.Sprintf("s3://%s/%s%s", f.directory.bucket.Name, f.directory.keyPrefix, f.relativePath)
}

func (f *s3File) pathInBucket() string {
	return fmt.Sprintf("%s%s", f.directory.keyPrefix, f.relativePath)
}
func (f *s3File) size() uint {
	return f._size
}

func (f *s3File) checksum() (string, error) {
	return f._checksum, nil
}

func (f *s3File) Open() error {
	rc, err := f.directory.bucket.GetReader(f.pathInBucket())
	if err != nil {
		return err
	}
	f.rc = rc
	return nil
}

func (f *s3File) Close() error {
	if f.rc == nil {
		return nil
	}
	return f.rc.Close()
}

func (f *s3File) Read(into []byte) (n int, err error) {
	return f.rc.Read(into)
}

func (f *s3File) Write(from []byte) (n int, err error) {
	if len(from) < multipart_threshold {
		return f.writeSimple(from)
	} else {
		return f.writeMultipart(from)
	}
}

func (f *s3File) writeSimple(from []byte) (n int, err error) {
	Log.Debug("initiating simple write of %+v to %s (%d bytes)", f, f.pathInBucket(), len(from))
	err = f.directory.bucket.PutReader(
		f.pathInBucket(),
		bytes.NewReader(from),
		int64(len(from)),
		"binary/octet-stream",
		s3.BucketOwnerFull,
	)
	if err != nil {
		return 0, err
	}
	return int(f.size()), nil
}

func (f *s3File) writeMultipart(from []byte) (n int, err error) {
	Log.Debug("initiating multipart write of %+v to %s (%d bytes)", f, f.pathInBucket(), len(from))
	multi, err := f.directory.bucket.InitMulti(
		f.pathInBucket(),
		"binary/octet-stream",
		s3.BucketOwnerFull,
	)
	if err != nil {
		return 0, fmt.Errorf("initmulti: %v", err)
	}
	Log.Debug("puttinf multiparts of %+v", f)
	parts, err := multi.PutAll(bytes.NewReader(from), multipart_part_size)
	if err != nil {
		return 0, fmt.Errorf("multi.putall: %v", err)
	}
	Log.Debug("completing %d multiparts of %+v", len(parts), f)
	if err = multi.Complete(parts); err != nil {
		return 0, fmt.Errorf("multi.complete: %v", err)
	}
	return int(f.size()), nil
}

func (f *s3File) streamMultipartWrite(from io.Reader) error {
	errs := make(chan error)
	parts := make(chan s3.Part)
	Log.Debug("initializing multipart upload for %+v", f)
	multi, err := f.directory.bucket.InitMulti(
		f.pathInBucket(),
		"binary/octet-stream",
		s3.BucketOwnerFull,
	)
	if err != nil {
		return fmt.Errorf("initmulti: %v", err)
	}

	Log.Debug("chunking input stream for %+v's multipart upload", f)
	waitgroup := &sync.WaitGroup{}
	eof := false
	i := 1 // s3 multipart uploads are 1-indexed
	for !eof {
		waitgroup.Add(1)

		buf := make([]byte, multipart_part_size)

		n, err := io.ReadFull(from, buf)
		if err == io.ErrUnexpectedEOF {
			buf = buf[:n]
			eof = true
		} else if err == io.EOF {
			eof = true
		} else if err != nil {
			// unknown err
			return err
		}
		Log.Debug("read %d bytes from stream and put it in buffer at %p", n, buf)

		go func(wg *sync.WaitGroup, mp *s3.Multi, b []byte, partNo int) {
			Log.Debug("starting upload worker for part no %d (%d bytes)", partNo, len(b))
			defer wg.Done()
			part, err := multi.PutPart(partNo, bytes.NewReader(b))
			if err != nil {
				errs <- err
			}
			Log.Debug("done with part no %d", partNo)
			parts <- part

		}(waitgroup, multi, buf, i)

		i += 1
	}
	Log.Debug("gathering parts")
	partsSlice := make([]s3.Part, 0)
	for len(partsSlice) < (i - 1) { // undo the 1-indexing of multipart uploads :\
		select {
		case err = <-errs:
			return err
		case part := <-parts:
			partsSlice = append(partsSlice, part)
			Log.Debug("got part %d (%d bytes) - at %d / %d", part.N, part.Size, len(partsSlice), (i - 1))
		}
	}

	if err = multi.Complete(partsSlice); err != nil {
		return fmt.Errorf("multi.complete: %v", err)
	}
	Log.Debug("done uploading %+v", f)
	return nil
}
