package s3sync

import (
	"fmt"
	"strings"
)

var (
	SkipChecksums bool
	DeleteMissing bool
)

func Sync(from, to string) error {
	fromDir, err := pathToDir(from)
	if err != nil {
		return err
	}
	toDir, err := pathToDir(to)
	if err != nil {
		return err
	}
	Log.Info("syncing %+v -> %+v", fromDir, toDir)
	return synchronize(fromDir, toDir)
}

func pathToDir(path string) (Directory, error) {
	if strings.HasPrefix(path, "s3://") {
		d, err := newS3Directory(path)
		if err != nil {
			return nil, err
		}
		return d, nil
	} else {
		if err := validate(path); err != nil {
			return nil, err
		}
		d := &localDirectory{path}
		return d, nil
	}
}

func synchronize(from, to Directory) error {
	Log.Debug("starting sync")
	fromC, err := from.Contents()
	if err != nil {
		return err
	}
	toC, err := to.Contents()
	if err != nil {
		return err
	}
	created, deleted, updated, _, err := computeDiff(fromC, toC)
	if err != nil {
		return err
	}
	if err := to.Clean(); err != nil {
		return err
	}
	if err := doCreates(to, created); err != nil {
		return err
	}
	if err := doUpdates(to, updated); err != nil {
		return err
	}
	if DeleteMissing {
		if err := doDeletes(to, deleted); err != nil {
			return err
		}
	}
	if err := to.Clean(); err != nil {
		return err
	}
	Log.Info("done synchronizing")
	return nil
}

func computeDiff(from, to map[string]file) (
	created, deleted, updated, unchanged map[string]file, err error) {
	Log.Info("computing diff of trees")
	created = make(map[string]file)
	deleted = make(map[string]file)
	updated = make(map[string]file)
	unchanged = make(map[string]file)
	seen := make(map[string]struct{})

	var (
		ok     bool
		toFile file
	)
	for path, fromFile := range from {
		seen[path] = struct{}{}

		toFile, ok = to[path]
		if !ok {
			Log.Debug("marking %s for creation", path)
			created[path] = fromFile
		} else {
			unch, err := fileUnchanged(fromFile, toFile)
			if err != nil {
				return nil, nil, nil, nil, err
			}
			if unch {
				Log.Debug("marking %s unchanged", path)
				unchanged[path] = fromFile
			} else {
				Log.Debug("marking %s for update", path)
				updated[path] = fromFile
			}
		}
	}
	for path, toFile := range to {
		if _, alreadySeen := seen[path]; !alreadySeen {
			Log.Debug("marking %s for delete", path)
			deleted[path] = toFile
		}
	}
	return created, deleted, updated, unchanged, nil
}

func doCreates(to Directory, creates map[string]file) error {
	Log.Info("doing creates")
	for path, file := range creates {
		file.Open()
		err := to.Create(path, file)
		if err != nil {
			return err
		}
		file.Close()
	}
	return nil
}

func doUpdates(to Directory, updates map[string]file) error {
	Log.Info("doing updates")
	for path, file := range updates {
		err := to.Update(path, file)
		if err != nil {
			return err
		}
	}
	return nil
}

func doDeletes(to Directory, deletes map[string]file) error {
	Log.Info("doing deletes")
	for path := range deletes {
		err := to.Delete(path)
		if err != nil {
			return err
		}
	}
	return nil
}

func fileUnchanged(from, to file) (bool, error) {
	Log.Debug("checking from %s vs to %s", from.fullPath(), to.fullPath())
	Log.Debug("size: from %d \t to %d", from.size(), to.size())
	if from.size() != to.size() {
		Log.Debug("size mismatch!")
		return false, nil
	}
	if !SkipChecksums {
		Log.Debug("checking checksums...")
		fchk, err := from.checksum()
		if err != nil {
			return false, fmt.Errorf("from.checksum(): %v", err)
		}
		tchk, err := to.checksum()
		if err != nil {
			return false, fmt.Errorf("to.checksum(): %v", err)
		}
		Log.Debug("checksums: from %s \t to %s", fchk, tchk)
		return fchk == tchk, nil
	}
	Log.Debug("looks good")
	return true, nil
}
