package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"net/smtp"
	"time"
)

var eslasticsearch_addr string

type ConfigStruct struct {
	ElasticSearchAddr string
	Alerts            []Alert
}

type Alert struct {
	Interval  float64
	Threshold float64
	Message   string
	QueryData json.RawMessage
}

func handle_alerts() {

	config_data := read_config_to_json()
	eslasticsearch_addr = config_data.ElasticSearchAddr
	for _, alert := range config_data.Alerts {
		go do_periodic_check(alert)
	}
}

func read_config_to_json() ConfigStruct {

	dat, err := ioutil.ReadFile("config.json")
	if err != nil {
		log.Fatalf("Error reading config file: %s", err)
	}
	var json_dat ConfigStruct
	if err := json.Unmarshal(dat, &json_dat); err != nil {
		log.Fatalf("Wrong json syntax: %s", err)
	}
	return json_dat
}

func do_periodic_check(alert Alert) {
	interval := time.Second * time.Duration(int(alert.Interval))
	for _ = range time.Tick(interval) {
		resp, err := http.Post(eslasticsearch_addr, "application/x-www-form-urlencoded", bytes.NewReader(alert.QueryData))
		if err != nil {
			log.Fatal("Failed to execute Elasticsearch query: %s ", alert.QueryData)
		}
		data, err := ioutil.ReadAll(resp.Body)
		resp.Body.Close()
		if err != nil {
			log.Fatal(err)
		}
		var json_data map[string]interface{}
		if err := json.Unmarshal(data, &json_data); err != nil {
			log.Fatalf("%s", err)
		}
		if hits_map, ok := json_data["hits"].(map[string]interface{}); ok {
			if total_hits, ok := hits_map["total"].(float64); ok {
				if total_hits > alert.Threshold {
					log.Println("Hits exceeded threshold: %s", alert.Message)
					go send_mail(alert.Message)
				}
			} else {
				log.Fatalf("Invalid data type for total_hits")
			}
		} else {
			log.Fatalf("Type assertion failed on hits_map")
		}
	}
}

func send_mail(msg string) {
	c, err := smtp.Dial("smtp.internal.justin.tv:25")
	if err != nil {
		log.Fatalf("%s", err)
	}
	c.Mail("systemsecurity@twitch.tv")
	c.Rcpt("security-alerts@twitchoncall.pagerduty.com")
	wc, err := c.Data()
	if err != nil {
		log.Fatalf("%s", err)
	}
	defer wc.Close()
	buf := bytes.NewBufferString(fmt.Sprintf("Subject: %s\r\n\r\n%s", "Dummy Alert", msg))
	if _, err = buf.WriteTo(wc); err != nil {
		log.Fatalf("%s", err)
	}
	log.Printf("Email sent")

}

func main() {
	log.SetFlags(0)
	log.Printf("hello world")
	handle_alerts()
	select {}
}
