package main

import (
	"net"

	"code.google.com/p/gopacket/layers"
)

type aggregatorKey [16]byte

type Aggregator struct {
	byteCache    map[aggregatorKey]uint64
	requestCache map[aggregatorKey]uint64
}

func NewAggregator() *Aggregator {
	return &Aggregator{
		byteCache:    make(map[aggregatorKey]uint64),
		requestCache: make(map[aggregatorKey]uint64),
	}
}

func (a *Aggregator) key(srcIP net.IP, srcPort uint16, dstIP net.IP, dstPort uint16, protocol layers.IPProtocol) aggregatorKey {
	var key [16]byte
	copy(key[0:4], []byte(srcIP)[0:4])
	copy(key[4:8], []byte(dstIP)[0:4])
	key[8] = byte(srcPort >> 8)
	key[9] = byte(srcPort & 0xff)
	key[10] = byte(dstPort >> 8)
	key[11] = byte(dstPort & 0xff)
	key[12] = byte(protocol)
	copy(key[13:16], []byte{0, 0, 0})
	return aggregatorKey(key)
}

func (a *Aggregator) unkey(key aggregatorKey) (srcIP net.IP, srcPort uint16, dstIP net.IP, dstPort uint16, protocol layers.IPProtocol) {
	srcIPBytes := make([]byte, 4)
	dstIPBytes := make([]byte, 4)
	copy(srcIPBytes[0:4], key[0:4])
	copy(dstIPBytes[0:4], key[4:8])
	srcIP = net.IP(srcIPBytes)
	dstIP = net.IP(dstIPBytes)
	srcPort = (uint16(key[8]) << 8) | uint16(key[9])
	dstPort = (uint16(key[10]) << 9) | uint16(key[11])
	protocol = layers.IPProtocol(key[12])
	return srcIP, srcPort, dstIP, dstPort, protocol
}

func (a *Aggregator) Add(packet *CapturedPacket) {
	key := a.key(packet.SourceIP, packet.SourcePort, packet.DestinationIP, packet.DestinationPort, packet.Protocol)
	a.byteCache[key] += uint64(packet.Length)
	a.requestCache[key] += 1
}

func (a *Aggregator) ForEach(f func(*CapturedPacketDigest)) {
	for k := range a.byteCache {
		byteCount := a.byteCache[k]
		requestCount := a.requestCache[k]
		srcIP, srcPort, dstIP, dstPort, protocol := a.unkey(k)
		p := &CapturedPacketDigest{
			Protocol:        protocol.String(),
			SFlowHost:       hostname,
			SourceIP:        srcIP.String(),
			SourcePort:      srcPort,
			DestinationIP:   dstIP.String(),
			DestinationPort: dstPort,
			Bytes:           byteCount,
			Packets:         requestCount,
		}
		f(p)
	}
}
